/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.web.rest.v2;

import static org.junit.Assert.assertEquals;

import org.json.JSONArray;
import org.json.JSONObject;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.opennms.core.test.MockLogAppender;
import org.opennms.core.test.OpenNMSJUnit4ClassRunner;
import org.opennms.core.test.db.annotations.JUnitTemporaryDatabase;
import org.opennms.core.test.rest.AbstractSpringJerseyRestTestCase;
import org.opennms.test.JUnitConfigurationEnvironment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.web.WebAppConfiguration;

@RunWith(OpenNMSJUnit4ClassRunner.class)
@WebAppConfiguration
@ContextConfiguration(locations={
        "classpath:/META-INF/opennms/applicationContext-soa.xml",
        "classpath:/META-INF/opennms/applicationContext-commonConfigs.xml",
        "classpath:/META-INF/opennms/applicationContext-minimal-conf.xml",
        "classpath:/META-INF/opennms/applicationContext-dao.xml",
        "classpath:/META-INF/opennms/applicationContext-mockConfigManager.xml",
        "classpath*:/META-INF/opennms/component-service.xml",
        "classpath*:/META-INF/opennms/component-dao.xml",
        "classpath:/META-INF/opennms/applicationContext-databasePopulator.xml",
        "classpath:/META-INF/opennms/mockEventIpcManager.xml",
        "file:src/main/webapp/WEB-INF/applicationContext-svclayer.xml",
        "file:src/main/webapp/WEB-INF/applicationContext-cxf-common.xml"
})
@JUnitConfigurationEnvironment(systemProperties = "org.opennms.timeseries.strategy=integration")
@JUnitTemporaryDatabase
public class SnmpInterfaceRestServiceIT extends AbstractSpringJerseyRestTestCase {
    private static final Logger LOG = LoggerFactory.getLogger(SnmpInterfaceRestServiceIT.class);

    public SnmpInterfaceRestServiceIT() {
        super(CXF_REST_V2_CONTEXT_PATH);
    }

    @Override
    protected void afterServletStart() throws Exception {
        MockLogAppender.setupLogging(true, "DEBUG");
    }

    @Test
    @JUnitTemporaryDatabase
    public void testAllEndPoints() throws Exception {
        sendPost("/monitoringLocations", "<location location-name=\"location1\" monitoring-area=\"location1\" priority=\"1\"/>", 201);
        String node1 = "<node type=\"A\" label=\"TestMachine1\" foreignSource=\"JUnit\" foreignId=\"TestMachine1\">" +
                "<location>location1</location>" +
                "<labelSource>H</labelSource>" +
                "<sysContact>The Owner</sysContact>" +
                "<sysDescription>" +
                "Darwin TestMachine 9.4.0 Darwin Kernel Version 9.4.0: Mon Jun  9 19:30:53 PDT 2008; root:xnu-1228.5.20~1/RELEASE_I386 i386" +
                "</sysDescription>" +
                "<sysLocation>DevJam</sysLocation>" +
                "<sysName>TestMachine1</sysName>" +
                "<sysObjectId>.1.3.6.1.4.1.8072.3.2.255</sysObjectId>" +
                "</node>";
        sendPost("/nodes", node1, 201);

        LOG.warn(sendRequest(GET, "/nodes/1/snmpinterfaces", 204));
        LOG.warn(sendRequest(GET, "/nodes/1/snmpinterfaces/6", 404));
        LOG.warn(sendRequest(GET, "/snmpinterfaces", 204));
        LOG.warn(sendRequest(GET, "/snmpinterfaces/1", 404));

        String snmpInterface1 = "<snmpInterface ifIndex=\"6\">" +
                "<ifAdminStatus>1</ifAdminStatus>" +
                "<ifDescr>en1</ifDescr>" +
                "<ifName>en1</ifName>" +
                "<ifOperStatus>1</ifOperStatus>" +
                "<ifSpeed>10000000</ifSpeed>" +
                "<ifType>6</ifType>" +
                "<netMask>255.255.255.0</netMask>" +
                "<physAddr>001e5271136d</physAddr>" +
                "</snmpInterface>";
        sendPost("/nodes/1/snmpinterfaces", snmpInterface1, 201, "/nodes/1/snmpinterfaces/6");

        LOG.warn(sendRequest(GET, "/snmpinterfaces", parseParamData("_s=ifIndex==6"), 200));
        LOG.warn(sendRequest(GET, "/snmpinterfaces", parseParamData("_s=node.label==*1"), 200));
        LOG.warn(sendRequest(GET, "/snmpinterfaces", parseParamData("_s=ifName==en1"), 200));

        JSONObject response = new JSONObject(sendRequest(GET, "/snmpinterfaces", 200));
        assertEquals(1, response.getInt("count"));
        JSONArray objects = response.getJSONArray("snmpInterface");
        JSONObject object = objects.getJSONObject(0);
        LOG.warn(sendRequest(GET, "/snmpinterfaces/" + object.getInt("id"), 200)); // By ID

        // add another node in a different location with a duplicate IP
        sendPost("/monitoringLocations", "<location location-name=\"location2\" monitoring-area=\"location2\" priority=\"1\"/>", 201);
        String node2 = "<node type=\"A\" label=\"TestMachine2\" foreignSource=\"JUnit\" foreignId=\"TestMachine2\">" +
                "<location>location2</location>" +
                "<labelSource>H</labelSource>" +
                "<sysContact>The Owner</sysContact>" +
                "<sysDescription>" +
                "Darwin TestMachine 9.4.0 Darwin Kernel Version 9.4.0: Mon Jun  9 19:30:53 PDT 2008; root:xnu-1228.5.20~1/RELEASE_I386 i386" +
                "</sysDescription>" +
                "<sysLocation>DevJam</sysLocation>" +
                "<sysName>TestMachine1</sysName>" +
                "<sysObjectId>.1.3.6.1.4.1.8072.3.2.255</sysObjectId>" +
                "</node>";
        sendPost("/nodes", node2, 201);

        String snmpInterface2 = "<snmpInterface ifIndex=\"6\">" +
                "<ifAdminStatus>1</ifAdminStatus>" +
                "<ifDescr>en1</ifDescr>" +
                "<ifName>en1</ifName>" +
                "<ifOperStatus>1</ifOperStatus>" +
                "<ifSpeed>10000000</ifSpeed>" +
                "<ifType>6</ifType>" +
                "<netMask>255.255.255.0</netMask>" +
                "<physAddr>001e5271136d</physAddr>" +
                "</snmpInterface>";
        sendPost("/nodes/2/snmpinterfaces", snmpInterface2, 201, "/nodes/2/snmpinterfaces/6");
        LOG.warn(sendRequest(GET, "/snmpinterfaces", 200));

        String jsonResponse = sendRequest(GET, "/snmpinterfaces", parseParamData("_s=ifIndex==6"), 200);
        response = new org.json.JSONObject(jsonResponse);
        assertEquals(2, response.getInt("count"));
    }

    @Test
    @JUnitTemporaryDatabase
    public void testMultipleIPInterfacesForSnmpInterface() throws Exception {
        sendPost("/monitoringLocations", "<location location-name=\"location1\" monitoring-area=\"location1\" priority=\"1\"/>", 201);
        String node1 = "<node type=\"A\" label=\"TestMachine1\" foreignSource=\"JUnit\" foreignId=\"TestMachine1\">" +
                "<location>location1</location>" +
                "<labelSource>H</labelSource>" +
                "<sysContact>The Owner</sysContact>" +
                "<sysDescription>" +
                "Darwin TestMachine 9.4.0 Darwin Kernel Version 9.4.0: Mon Jun  9 19:30:53 PDT 2008; root:xnu-1228.5.20~1/RELEASE_I386 i386" +
                "</sysDescription>" +
                "<sysLocation>DevJam</sysLocation>" +
                "<sysName>TestMachine1</sysName>" +
                "<sysObjectId>.1.3.6.1.4.1.8072.3.2.255</sysObjectId>" +
                "</node>";
        sendPost("/nodes", node1, 201);

        String snmpInterface = "<snmpInterface ifIndex=\"6\">" +
                "<ifAdminStatus>1</ifAdminStatus>" +
                "<ifDescr>en1</ifDescr>" +
                "<ifName>en1</ifName>" +
                "<ifOperStatus>1</ifOperStatus>" +
                "<ifSpeed>10000000</ifSpeed>" +
                "<ifType>6</ifType>" +
                "<netMask>255.255.255.0</netMask>" +
                "<physAddr>001e5271136d</physAddr>" +
                "</snmpInterface>";

        sendPost("/nodes/1/snmpinterfaces", snmpInterface, 201, "/nodes/1/snmpinterfaces/6");

        String ipInterface1 = "<ipInterface snmpPrimary=\"P\">" +
                "<ipAddress>10.10.10.10</ipAddress>" +
                "<nodeId>1</nodeId>" +
                "<hostName>TestMachine1</hostName>" +
                "<ifIndex>6</ifIndex>" +
                "</ipInterface>";
        sendPost("/nodes/1/ipinterfaces", ipInterface1, 201);

        String ipInterface2 = "<ipInterface snmpPrimary=\"P\">" +
                "<ipAddress>10.10.10.11</ipAddress>" +
                "<nodeId>1</nodeId>" +
                "<hostName>TestMachine1</hostName>" +
                "<ifIndex>6</ifIndex>" +
                "</ipInterface>";
        sendPost("/nodes/1/ipinterfaces", ipInterface2, 201);

        JSONObject response = new JSONObject(sendRequest(GET, "/snmpinterfaces", 200));
        assertEquals(1, response.getInt("count"));

    }
}