/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.karaf.extender;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.karaf.features.FeaturesService;
import org.apache.karaf.features.Repository;
import org.apache.karaf.kar.KarService;
import org.opennms.karaf.extender.Feature;
import org.opennms.karaf.extender.KarafExtender;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KarDependencyHandler
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(KarDependencyHandler.class);
    private static final int KAR_LIST_SLEEP_MS = 5000;
    private final List<Feature> features;
    private final KarService karService;
    private final FeaturesService featuresService;
    private final KarafExtender.ExtenderStatus extenderStatus;
    private static final String FEATURE_CONFIG_FILE = "features.cfg";
    private static final String KAR_STORAGE = System.getProperty("karaf.data") + File.separator + "kar";

    public KarDependencyHandler(List<Feature> features, KarService karService, FeaturesService featuresService, KarafExtender.ExtenderStatus extenderStatus) {
        this.features = Objects.requireNonNull(features);
        this.karService = Objects.requireNonNull(karService);
        this.featuresService = Objects.requireNonNull(featuresService);
        this.extenderStatus = Objects.requireNonNull(extenderStatus);
    }

    @Override
    public void run() {
        Set allKarDependencies = this.features.stream().map(Feature::getKarDependency).collect(Collectors.toSet());
        HashSet karsToWaitFor = new HashSet(allKarDependencies);
        while (true) {
            try {
                this.extenderStatus.info("Waiting on {}", karsToWaitFor);
                karsToWaitFor.removeAll(this.karService.list());
                if (karsToWaitFor.isEmpty()) {
                    break;
                }
            }
            catch (Exception e) {
                this.extenderStatus.warn("Enumerating installed .kar files failed. Will retry in {}ms.", 5000, e);
            }
            try {
                Thread.sleep(5000L);
            }
            catch (InterruptedException e) {
                LOG.info("Interrupted. Stopping thread.");
                return;
            }
        }
        LOG.info("All .kar dependencies are ready now.");
        HashSet<URI> availableFeatureUris = new HashSet<URI>();
        try {
            for (Repository repository : this.featuresService.listRepositories()) {
                availableFeatureUris.add(repository.getURI());
            }
        }
        catch (Exception e) {
            this.extenderStatus.warn("Failed to retrieve feature repository details. Assuming there are no feature repositories installed.", e);
        }
        HashSet<URI> missingFeatureUris = new HashSet<URI>();
        for (String karDependency : allKarDependencies) {
            missingFeatureUris.addAll(this.getFeaturesUrisForKar(karDependency));
        }
        missingFeatureUris.removeAll(availableFeatureUris);
        if (missingFeatureUris.isEmpty()) {
            LOG.debug("No missing feature repositories.");
        } else {
            LOG.info("Installing feature repositories: {}", missingFeatureUris);
            for (URI featureUri : missingFeatureUris) {
                try {
                    this.featuresService.addRepository(featureUri);
                }
                catch (Exception e) {
                    this.extenderStatus.error("Failed to install feature repository: {}", featureUri, e);
                }
            }
        }
        Set featuresToInstall = this.features.stream().map(Feature::toInstallString).collect(Collectors.toSet());
        try {
            LOG.info("Installing features: {}", featuresToInstall);
            this.featuresService.installFeatures(featuresToInstall, EnumSet.noneOf(FeaturesService.Option.class));
        }
        catch (Exception e) {
            this.extenderStatus.error("Failed to install one or more features", e);
        }
        this.extenderStatus.karsDone(String.format("%s KAR features installed", featuresToInstall.size()));
    }

    private List<URI> getFeaturesUrisForKar(String kar) {
        Path featureConfig = Paths.get(KAR_STORAGE, kar, FEATURE_CONFIG_FILE);
        File featureConfigFile = featureConfig.toFile();
        if (!featureConfigFile.isFile()) {
            LOG.debug("Kar '{}' is installed, but the feature configuration is not yet written. Waiting up-to 30 seconds for it to show up...", (Object)kar);
            try {
                for (int i = 30; i > 0 && !featureConfigFile.isFile(); --i) {
                    Thread.sleep(1000L);
                }
            }
            catch (InterruptedException e) {
                LOG.info("Interrupted while waiting for {}. Assuming no feature repositories are used.", (Object)featureConfigFile);
                return Collections.emptyList();
            }
        }
        try {
            LOG.debug("Reading feature repository list for kar '{}' in: {}", (Object)kar, (Object)featureConfig);
            return Files.readAllLines(featureConfig).stream().map(URI::create).collect(Collectors.toList());
        }
        catch (IOException e) {
            this.extenderStatus.warn("Cannot read feature repository list for kar '{}' in: {}. Assuming no feature repositories are used.", kar, featureConfig, e);
            return Collections.emptyList();
        }
    }
}

