/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.netmgt.flows.elastic;

import com.codahale.metrics.Meter;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import io.opentracing.Scope;
import io.opentracing.Tracer;
import io.opentracing.util.GlobalTracer;
import java.io.IOException;
import java.time.Instant;
import java.time.temporal.TemporalAccessor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TimerTask;
import java.util.concurrent.locks.ReentrantLock;
import org.opennms.core.tracing.api.TracerRegistry;
import org.opennms.distributed.core.api.Identity;
import org.opennms.features.elastic.client.ElasticRestClient;
import org.opennms.features.elastic.client.model.BulkRequest;
import org.opennms.features.elastic.client.model.BulkResponse;
import org.opennms.features.jest.client.bulk.FailedItem;
import org.opennms.features.jest.client.index.IndexStrategy;
import org.opennms.features.jest.client.template.IndexSettings;
import org.opennms.integration.api.v1.flows.Flow;
import org.opennms.integration.api.v1.flows.FlowException;
import org.opennms.integration.api.v1.flows.FlowRepository;
import org.opennms.netmgt.flows.elastic.FlowDocument;
import org.opennms.netmgt.flows.elastic.PersistenceException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ElasticFlowRepository
implements FlowRepository {
    public static final String TRACER_FLOW_MODULE = "ElasticFlow";
    private static final Logger LOG = LoggerFactory.getLogger(ElasticFlowRepository.class);
    private static final String INDEX_NAME = "netflow";
    private final ElasticRestClient client;
    private final IndexStrategy indexStrategy;
    private final Meter flowsPersistedMeter;
    private final Timer logPersistingTimer;
    private final Identity identity;
    private final TracerRegistry tracerRegistry;
    private final IndexSettings indexSettings;
    private int bulkSize = 1000;
    private int bulkFlushMs = 500;
    private final Map<Thread, FlowBulk> flowBulks = Maps.newConcurrentMap();
    private java.util.Timer flushTimer;

    public ElasticFlowRepository(MetricRegistry metricRegistry, ElasticRestClient elasticRestClient, IndexStrategy indexStrategy, Identity identity, TracerRegistry tracerRegistry, IndexSettings indexSettings) {
        this.client = Objects.requireNonNull(elasticRestClient);
        this.indexStrategy = Objects.requireNonNull(indexStrategy);
        this.identity = identity;
        this.tracerRegistry = tracerRegistry;
        this.indexSettings = Objects.requireNonNull(indexSettings);
        this.flowsPersistedMeter = metricRegistry.meter("flowsPersisted");
        this.logPersistingTimer = metricRegistry.timer("logPersisting");
        this.startTimer();
    }

    public ElasticFlowRepository(MetricRegistry metricRegistry, ElasticRestClient elasticRestClient, IndexStrategy indexStrategy, Identity identity, TracerRegistry tracerRegistry, IndexSettings indexSettings, int bulkSize, int bulkFlushMs) {
        this(metricRegistry, elasticRestClient, indexStrategy, identity, tracerRegistry, indexSettings);
        this.bulkSize = bulkSize;
        this.bulkFlushMs = bulkFlushMs;
    }

    private void startTimer() {
        if (this.flushTimer != null) {
            return;
        }
        if (this.bulkFlushMs > 0) {
            int delay = Math.max(1, this.bulkFlushMs / 2);
            this.flushTimer = new java.util.Timer("ElasticFlowRepositoryFlush");
            this.flushTimer.scheduleAtFixedRate(new TimerTask(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void run() {
                    long currentTimeMillis = System.currentTimeMillis();
                    for (Map.Entry<Thread, FlowBulk> entry : ElasticFlowRepository.this.flowBulks.entrySet()) {
                        FlowBulk flowBulk = entry.getValue();
                        if (currentTimeMillis - flowBulk.lastPersist <= (long)ElasticFlowRepository.this.bulkFlushMs || !flowBulk.lock.tryLock()) continue;
                        try {
                            if (flowBulk.documents.size() <= 0) continue;
                            try {
                                ElasticFlowRepository.this.persistBulk(flowBulk.documents);
                                flowBulk.lastPersist = currentTimeMillis;
                            }
                            catch (Throwable t) {
                                LOG.error("An error occurred while flushing one or more bulks in ElasticFlowRepository.", t);
                            }
                        }
                        finally {
                            flowBulk.lock.unlock();
                        }
                    }
                }
            }, delay, (long)delay);
        } else {
            this.flushTimer = null;
        }
    }

    private void stopTimer() {
        if (this.flushTimer != null) {
            this.flushTimer.cancel();
            this.flushTimer = null;
        }
    }

    public void persist(Collection<? extends Flow> flows) throws FlowException {
        FlowBulk flowBulk = this.flowBulks.computeIfAbsent(Thread.currentThread(), thread -> new FlowBulk());
        flowBulk.lock.lock();
        try {
            flows.stream().map(FlowDocument::from).forEach(flowBulk.documents::add);
            if (flowBulk.documents.size() >= this.bulkSize) {
                this.persistBulk(flowBulk.documents);
                flowBulk.lastPersist = System.currentTimeMillis();
            }
        }
        finally {
            flowBulk.lock.unlock();
        }
    }

    private void persistBulk(List<FlowDocument> bulk) throws FlowException {
        LOG.debug("Persisting {} flow documents.", (Object)bulk.size());
        Tracer tracer = this.getTracer();
        try (Timer.Context ctx = this.logPersistingTimer.time();
             Scope scope = tracer.buildSpan(TRACER_FLOW_MODULE).startActive(true);){
            scope.span().setTag("thread", Thread.currentThread().getName());
            BulkRequest bulkRequest = new BulkRequest();
            for (FlowDocument flowDocument : bulk) {
                String index = this.indexStrategy.getIndex(this.indexSettings, INDEX_NAME, (TemporalAccessor)Instant.ofEpochMilli(flowDocument.getTimestamp()));
                bulkRequest.index(index, null, (Object)flowDocument);
            }
            try {
                BulkResponse response = this.client.executeBulk(bulkRequest);
                if (response.hasErrors()) {
                    List<FailedItem<FlowDocument>> failedItems = ElasticFlowRepository.getFailedItems(response);
                    throw new PersistenceException(response.getErrors(), failedItems);
                }
            }
            catch (IOException ex) {
                LOG.error("An error occurred while executing the given request: {}", (Object)ex.getMessage(), (Object)ex);
                throw new FlowException(ex.getMessage(), (Throwable)ex);
            }
            this.flowsPersistedMeter.mark((long)bulk.size());
            bulk.clear();
        }
    }

    private static List<FailedItem<FlowDocument>> getFailedItems(BulkResponse response) {
        ArrayList<FailedItem<FlowDocument>> failedItems = new ArrayList<FailedItem<FlowDocument>>();
        List bulkFailedItems = response.getFailedItems();
        for (int i = 0; i < bulkFailedItems.size(); ++i) {
            BulkResponse.BulkItemResponse item = (BulkResponse.BulkItemResponse)bulkFailedItems.get(i);
            FlowDocument doc = new FlowDocument();
            doc.setConvoKey("unknown-" + i);
            FailedItem failedItem = new FailedItem(i, (Object)doc, new Exception(item.getError() != null ? item.getError() : "Unknown error"));
            failedItems.add((FailedItem<FlowDocument>)failedItem);
        }
        return failedItems;
    }

    public Identity getIdentity() {
        return this.identity;
    }

    public TracerRegistry getTracerRegistry() {
        return this.tracerRegistry;
    }

    public void start() {
        if (this.tracerRegistry != null && this.identity != null) {
            this.tracerRegistry.init(this.identity.getId());
        }
        this.startTimer();
    }

    public void stop() throws FlowException {
        this.stopTimer();
        for (FlowBulk flowBulk : this.flowBulks.values()) {
            this.persistBulk(flowBulk.documents);
        }
    }

    private Tracer getTracer() {
        if (this.tracerRegistry != null) {
            return this.tracerRegistry.getTracer();
        }
        return GlobalTracer.get();
    }

    public int getBulkSize() {
        return this.bulkSize;
    }

    public void setBulkSize(int bulkSize) {
        this.bulkSize = bulkSize;
    }

    public int getBulkFlushMs() {
        return this.bulkFlushMs;
    }

    public void setBulkFlushMs(int bulkFlushMs) {
        this.bulkFlushMs = bulkFlushMs;
        this.stopTimer();
        this.startTimer();
    }

    private class FlowBulk {
        private List<FlowDocument> documents;
        private ReentrantLock lock;
        private long lastPersist;

        public FlowBulk() {
            this.documents = Lists.newArrayListWithCapacity((int)ElasticFlowRepository.this.bulkSize);
            this.lock = new ReentrantLock();
            this.lastPersist = 0L;
        }
    }
}

