/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.netmgt.timeseries.samplewrite;

import com.codahale.metrics.Gauge;
import com.codahale.metrics.Meter;
import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleMath;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.lmax.disruptor.EventTranslatorOneArg;
import com.lmax.disruptor.ExceptionHandler;
import com.lmax.disruptor.FatalExceptionHandler;
import com.lmax.disruptor.RingBuffer;
import com.lmax.disruptor.Sequence;
import com.lmax.disruptor.WorkHandler;
import com.lmax.disruptor.WorkerPool;
import com.swrve.ratelimitedlogger.RateLimitedLog;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Named;
import org.opennms.integration.api.v1.timeseries.Sample;
import org.opennms.integration.api.v1.timeseries.TimeSeriesStorage;
import org.opennms.netmgt.timeseries.TimeseriesStorageManager;
import org.opennms.netmgt.timeseries.samplewrite.SampleBatchEvent;
import org.opennms.netmgt.timeseries.samplewrite.TimeseriesWriter;
import org.opennms.netmgt.timeseries.stats.StatisticsCollector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;

public class RingBufferTimeseriesWriter
implements TimeseriesWriter,
WorkHandler<SampleBatchEvent>,
DisposableBean {
    private static final Logger LOG = LoggerFactory.getLogger(RingBufferTimeseriesWriter.class);
    private static final RateLimitedLog RATE_LIMITED_LOGGER = RateLimitedLog.withRateLimit((Logger)LOG).maxRate(5).every(Duration.ofSeconds(30L)).build();
    private static final Duration STORAGE_GET_WARNING_DURATION = Duration.ofSeconds(5L);
    private static final Duration DESTROY_GRACE_PERIOD = Duration.ofSeconds(30L);
    private WorkerPool<SampleBatchEvent> workerPool;
    private ExecutorService executor;
    private RingBuffer<SampleBatchEvent> ringBuffer;
    private final int ringBufferSize;
    private final int numWriterThreads;
    private final Meter droppedSamples;
    private final Timer sampleWriteTsTimer;
    private TimeseriesStorageManager storage;
    private StatisticsCollector stats;
    private final AtomicLong numEntriesOnRingBuffer = new AtomicLong();
    private final AtomicBoolean readyToRockAndRoll = new AtomicBoolean(false);
    private final AtomicBoolean thePartyIsOver = new AtomicBoolean(false);
    private static final EventTranslatorOneArg<SampleBatchEvent, List<Sample>> TRANSLATOR = (event, sequence, samples) -> event.setSamples((List<Sample>)samples);

    @Inject
    public RingBufferTimeseriesWriter(TimeseriesStorageManager storage, StatisticsCollector stats, @Named(value="timeseries.ring_buffer_size") Integer ringBufferSize, @Named(value="timeseries.writer_threads") Integer numWriterThreads, @Named(value="timeseriesMetricRegistry") MetricRegistry registry) {
        Preconditions.checkArgument((ringBufferSize > 0 ? 1 : 0) != 0, (Object)"ringBufferSize must be positive");
        Preconditions.checkArgument((boolean)DoubleMath.isMathematicalInteger((double)(Math.log(ringBufferSize.intValue()) / Math.log(2.0))), (Object)"ringBufferSize must be a power of two");
        Preconditions.checkArgument((numWriterThreads > 0 ? 1 : 0) != 0, (Object)"numWriterThreads must be positive");
        Preconditions.checkNotNull((Object)registry, (Object)"metric registry");
        this.storage = Objects.requireNonNull(storage);
        this.stats = Objects.requireNonNull(stats);
        this.ringBufferSize = ringBufferSize;
        this.numWriterThreads = numWriterThreads;
        this.numEntriesOnRingBuffer.set(0L);
        registry.register(MetricRegistry.name((String)"ring-buffer", (String[])new String[]{"size"}), (Metric)((Gauge)this.numEntriesOnRingBuffer::get));
        registry.register(MetricRegistry.name((String)"ring-buffer", (String[])new String[]{"max-size"}), (Metric)((Gauge)() -> this.ringBufferSize));
        this.droppedSamples = registry.meter(MetricRegistry.name((String)"ring-buffer", (String[])new String[]{"dropped-samples"}));
        this.sampleWriteTsTimer = registry.timer("samples.write.ts");
        LOG.debug("Using ring_buffer_size: {}", (Object)this.ringBufferSize);
        this.setUpWorkerPool();
    }

    private void setUpWorkerPool() {
        ThreadFactory namedThreadFactory = new ThreadFactoryBuilder().setNameFormat("TimeseriesWriter-Consumer-%d").build();
        this.executor = Executors.newCachedThreadPool(namedThreadFactory);
        WorkHandler[] handlers = new WorkHandler[this.numWriterThreads];
        for (int i = 0; i < this.numWriterThreads; ++i) {
            handlers[i] = this;
        }
        this.ringBuffer = RingBuffer.createMultiProducer(SampleBatchEvent::new, (int)this.ringBufferSize);
        this.workerPool = new WorkerPool(this.ringBuffer, this.ringBuffer.newBarrier(new Sequence[0]), (ExceptionHandler)new FatalExceptionHandler(), handlers);
        this.ringBuffer.addGatingSequences(this.workerPool.getWorkerSequences());
        this.workerPool.start((Executor)this.executor);
        this.readyToRockAndRoll.set(true);
    }

    @Override
    public void destroy() {
        this.readyToRockAndRoll.set(false);
        if (this.workerPool != null) {
            Instant start = Instant.now();
            LOG.info("destroy(): Draining and halting the time series worker pool. Entries in ring buffer: {}", (Object)this.numEntriesOnRingBuffer.get());
            Thread destroyStatusThread = new Thread(() -> {
                while (this.numEntriesOnRingBuffer.get() != 0L && Duration.between(start, Instant.now()).compareTo(DESTROY_GRACE_PERIOD) < 0) {
                    LOG.info("destroy() in progress. Entries left in ring buffer to drain: {}", (Object)this.numEntriesOnRingBuffer.get());
                    try {
                        Thread.sleep(5000L);
                    }
                    catch (InterruptedException e) {
                        LOG.info("Apparently my work is done here. Entries in ring buffer: {}", (Object)this.numEntriesOnRingBuffer.get());
                        break;
                    }
                }
                if (this.numEntriesOnRingBuffer.get() != 0L) {
                    LOG.warn("destroy(): WorkerPool does not want to cooperate, forcing cooperation");
                    this.thePartyIsOver.set(true);
                    this.executor.shutdownNow();
                }
            }, this.getClass().getSimpleName() + "-destroy-status");
            destroyStatusThread.start();
            this.workerPool.drainAndHalt();
            LOG.info("Completed draining ring buffer entries (current size {}). Worker pool is halted. Took {}.", (Object)this.numEntriesOnRingBuffer.get(), (Object)Duration.between(start, Instant.now()));
            destroyStatusThread.interrupt();
        }
    }

    @Override
    public void insert(List<Sample> samples) {
        if (!this.readyToRockAndRoll.get()) {
            this.insertDrop(samples, "We are not ready to rock and roll");
            return;
        }
        if (!this.ringBuffer.tryPublishEvent(TRANSLATOR, samples)) {
            this.insertDrop(samples, "The ring buffer is full");
            return;
        }
        this.numEntriesOnRingBuffer.incrementAndGet();
    }

    private void insertDrop(final List<Sample> samples, String message) {
        RATE_LIMITED_LOGGER.error(message + ". {} samples associated with resource ids {} will be dropped.", (Object)samples.size(), new Object(){

            public String toString() {
                return samples.stream().map(s -> s.getMetric().getFirstTagByKey("resourceId").getValue()).distinct().collect(Collectors.joining(", "));
            }
        });
        this.droppedSamples.mark((long)samples.size());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void onEvent(SampleBatchEvent event) {
        if (this.thePartyIsOver.get()) {
            return;
        }
        try (Timer.Context context = this.sampleWriteTsTimer.time();){
            Instant start = Instant.now();
            TimeSeriesStorage timeSeriesStorage = this.storage.get();
            Duration getDuration = Duration.between(start, Instant.now());
            if (getDuration.compareTo(STORAGE_GET_WARNING_DURATION) > 0) {
                RATE_LIMITED_LOGGER.warn("storage.get() took an excessive amount of time, {}, and returned: {}", (Object)getDuration, (Object)timeSeriesStorage);
            }
            if (timeSeriesStorage == null) {
                RATE_LIMITED_LOGGER.error("There is no available TimeSeriesStorage implementation. {} samples will be lost.", (Object)event.getSamples().size());
            } else {
                timeSeriesStorage.store(event.getSamples());
                this.stats.record(event.getSamples());
            }
        }
        catch (Throwable t) {
            RATE_LIMITED_LOGGER.error("An error occurred while inserting samples. Up to {} samples may be lost: {}: {}", new Object[]{event.getSamples().size(), t.getClass().getSimpleName(), t.getMessage(), t});
        }
        finally {
            event.setSamples(null);
        }
        this.numEntriesOnRingBuffer.decrementAndGet();
    }

    public void setTimeSeriesStorage(TimeseriesStorageManager timeseriesStorage) {
        this.storage = timeseriesStorage;
    }

    public void setStats(StatisticsCollector stats) {
        this.stats = stats;
    }
}

