/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.dao.jaxb;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.opennms.core.xml.AbstractJaxbConfigDao;
import org.opennms.netmgt.config.surveillanceViews.SurveillanceViewConfiguration;
import org.opennms.netmgt.config.surveillanceViews.View;
import org.opennms.netmgt.dao.api.SurveillanceViewConfigDao;

/**
 * <p>DefaultSurveillanceViewConfigDao class.</p>
 *
 * @author ranger
 * @version $Id: $
 */
public class DefaultSurveillanceViewConfigDao extends AbstractJaxbConfigDao<SurveillanceViewConfiguration, SurveillanceViewConfig> implements SurveillanceViewConfigDao {
    /**
     * <p>Constructor for DefaultSurveillanceViewConfigDao.</p>
     */
    public DefaultSurveillanceViewConfigDao() {
        super(SurveillanceViewConfiguration.class, "surveillance view configuration");
    }
    
    /** {@inheritDoc} */
    @Override
    public SurveillanceViewConfig translateConfig(SurveillanceViewConfiguration config) {
        return new SurveillanceViewConfig(config);
    }
    
    /** {@inheritDoc} */
    @Override
    public View getView(String viewName) {
        return getContainer().getObject().getViewsMap().get(viewName);
    }
    
    /**
     * Use this method to get the generated Views class generated by the XSD.
     *
     * @return a {@link org.opennms.netmgt.config.surveillanceViews.Views} object.
     */
    @Override
    public List<View> getViews() {
        return getContainer().getObject().getConfig().getViews();
    }

    /**
     * Use this method to get a Map of view names to marshalled classes based on the generated View class
     * from the XSD.
     *
     * @return <code>Map</> of View classes.
     */
    public Map<String, View> getViewMap() {
        return Collections.unmodifiableMap(getContainer().getObject().getViewsMap());
    }

    /**
     * <p>getDefaultView</p>
     *
     * @return a {@link org.opennms.netmgt.config.surveillanceViews.View} object.
     */
    @Override
    public View getDefaultView() {
        String defaultView = getContainer().getObject().getConfig().getDefaultView();
        return getView(defaultView);
    }

}
