/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.features.topology.plugins.topo.asset.layers;

import java.util.List;
import java.util.stream.Collectors;

import org.opennms.netmgt.model.OnmsNode;

/**
 * Generates ids for each {@link org.opennms.features.graphml.model.GraphMLNode} in the generated graph.
 * The {@link IdGenerator} is necessary, because due to the nature of Assets.
 * Let's assume a hierarchy "Region -> Building" should be created and two nodes from different Regions,
 * but the same building, either two or only one {@link org.opennms.features.graphml.model.GraphMLNode} must be drawn.
 *
 * The {@link IdGenerator} allows to consider the hierarchy an {@link org.opennms.features.graphml.model.GraphMLNode} is added into.
 *
 * Example:
 * <ul>
 *     <li>Node 1, Pittsboro, Building 1</li>
 *     <li>Node 2, Berlin, Building 1</li>
 * </ul>
 *
 * @author mvrueden
 */
public interface IdGenerator {

    String generateId(List<Layer> processedLayers, OnmsNode currentNode, String currentId);

    /**
     * The id generated by this generator considers hierarchy by simply concatenating the parent ids with the <code>currentId</code> (generated by {@link NodeDecorator#getId(Object)}).
     */
    IdGenerator HIERARCHY = new IdGenerator() {
        @Override
        public String generateId(List<Layer> processedLayers, OnmsNode currentNode, String currentId) {
            List<String> collectedValues = processedLayers.stream().map(l -> l.getItemProvider().getItem(currentNode).toString()).collect(Collectors.toList());
            collectedValues.add(currentId);
            return collectedValues.stream().collect(Collectors.joining("."));
        }
    };

    /**
     * Does not consider hierarchy. Simply returns the id generated by {@link NodeDecorator#getId(Object)} (<code>currentId</code>).
     */
    IdGenerator SIMPLE = new IdGenerator() {
        @Override
        public String generateId(List<Layer> processedLayers, OnmsNode currentNode, String currentId) {
            return currentId;
        }
    };
}
