/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.telemetry.protocols.sflow.parser.proto.flows;

import java.util.Optional;

import org.bson.BsonWriter;
import org.opennms.netmgt.telemetry.listeners.utils.BufferUtils;
import org.opennms.netmgt.telemetry.protocols.sflow.parser.SampleDatagramEnrichment;
import org.opennms.netmgt.telemetry.protocols.sflow.parser.InvalidPacketException;
import org.opennms.netmgt.telemetry.protocols.sflow.parser.SampleDatagramVisitor;
import org.opennms.netmgt.telemetry.protocols.sflow.parser.proto.Array;

import com.google.common.base.MoreObjects;

import io.netty.buffer.ByteBuf;

// struct counters_sample_expanded {
//    unsigned int sequence_number;   /* Incremented with each counter sample
//                                       generated by this source_id
//                                       Note: If the agent resets any of the
//                                             counters then it must also
//                                             reset the sequence_number.
//                                             In the case of ifIndex-based
//                                             source_id's the sequence
//                                             number must be reset each time
//                                             ifCounterDiscontinuityTime
//                                             changes. */
//    sflow_data_source_expanded source_id; /* sFlowDataSource */
//    counter_record counters<>;      /* Counters polled for this source */
// };

public class CountersSampleExpanded implements SampleData {
    public final long sequence_number;
    public final SFlowDataSourceExpanded source_id;
    public final Array<CounterRecord> counters;

    public CountersSampleExpanded(final ByteBuf buffer) throws InvalidPacketException {
        this.sequence_number = BufferUtils.uint32(buffer);
        this.source_id = new SFlowDataSourceExpanded(buffer);
        this.counters = new Array(buffer, Optional.empty(), CounterRecord::new);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("sequence_number", this.sequence_number)
                .add("source_id", this.source_id)
                .add("counters", this.counters)
                .toString();
    }

    @Override
    public void writeBson(final BsonWriter bsonWriter, final SampleDatagramEnrichment enr) {
        bsonWriter.writeStartDocument();
        bsonWriter.writeInt64("sequence_number", this.sequence_number);

        bsonWriter.writeName("source_id");
        this.source_id.writeBson(bsonWriter, enr);

        bsonWriter.writeStartDocument("counters");
        for (final CounterRecord counterRecord : this.counters) {
            bsonWriter.writeName(counterRecord.dataFormat.toId());
            counterRecord.writeBson(bsonWriter, enr);
        }
        bsonWriter.writeEndDocument();

        bsonWriter.writeEndDocument();
    }

    @Override
    public void visit(final SampleDatagramVisitor visitor) {
        visitor.accept(this);
    }
}
