/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennnms.netmgt.collectd.prometheus;

import static com.github.tomakehurst.wiremock.client.WireMock.aResponse;
import static com.github.tomakehurst.wiremock.client.WireMock.get;
import static com.github.tomakehurst.wiremock.client.WireMock.stubFor;
import static com.github.tomakehurst.wiremock.client.WireMock.urlEqualTo;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.hasSize;
import static org.junit.Assert.assertEquals;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.opennms.core.collection.test.CollectionSetUtils;
import org.opennms.core.utils.InetAddressUtils;
import org.opennms.netmgt.collectd.prometheus.PrometheusCollector;
import org.opennms.netmgt.collection.api.CollectionAgent;
import org.opennms.netmgt.collection.api.CollectionException;
import org.opennms.netmgt.collection.api.CollectionSet;
import org.opennms.netmgt.collection.api.ResourceTypeMapper;
import org.opennms.netmgt.collection.core.DefaultCollectionAgent;
import org.opennms.netmgt.collection.support.IndexStorageStrategy;
import org.opennms.netmgt.collection.support.PersistAllSelectorStrategy;
import org.opennms.netmgt.config.datacollection.PersistenceSelectorStrategy;
import org.opennms.netmgt.config.datacollection.ResourceType;
import org.opennms.netmgt.config.datacollection.StorageStrategy;
import org.opennms.netmgt.config.prometheus.Collection;
import org.opennms.netmgt.config.prometheus.Group;
import org.opennms.netmgt.config.prometheus.NumericAttribute;
import org.opennms.netmgt.config.prometheus.StringAttribute;
import org.opennms.netmgt.dao.prometheus.PrometheusDataCollectionConfigDao;
import org.opennms.netmgt.dao.api.IpInterfaceDao;
import org.opennms.netmgt.model.OnmsIpInterface;
import org.opennms.netmgt.model.OnmsNode;
import org.springframework.transaction.PlatformTransactionManager;

import com.github.tomakehurst.wiremock.core.WireMockConfiguration;
import com.github.tomakehurst.wiremock.junit.WireMockRule;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;

/**
 * Used to verify that the expected CollectionSet is generated by parsing
 * some know HTTP response with a programmatically defined configuration.
 *
 * These tests were built while developing the model and a bit verbose
 *  - could be simplified by making them "data driven".
 *
 * @author jwhite
 */
public class PrometheusCollectorIT {

    private PrometheusCollector collector = new PrometheusCollector();

    @Rule
    public WireMockRule wireMockRule = new WireMockRule(WireMockConfiguration.wireMockConfig()
            .withRootDirectory(Paths.get("src", "test", "resources").toString())
            .dynamicPort());

    @Before
    public void setUp() {
        stubFor(get(urlEqualTo("/metrics"))
                .willReturn(aResponse()
                        .withHeader("Content-Type", "Content-Type: text/plain; version=0.0.4")
                        .withBodyFile("linux.metrics")));
        stubFor(get(urlEqualTo("/nephron_on_flink"))
                .willReturn(aResponse()
                        .withHeader("Content-Type", "Content-Type: text/plain; version=0.0.4")
                        .withBodyFile("flink.metrics")));
    }

    @Test
    public void canCollectFlinkTaskManagerMetrics() {
        // Define our configuration
        Collection collection = new Collection();
        Group flinkTask = new Group();
        flinkTask.setName("flink-task-manager");
        flinkTask.setFilterExp("name matches 'flink_taskmanager_Status_.*'");
        flinkTask.setGroupByExp("labels[tm_id]");
        flinkTask.setResourceType("flinkTaskManager");

        NumericAttribute attribute = new NumericAttribute();
        attribute.setAliasExp("name.substring('flink_taskmanager_Status_'.length())");
        flinkTask.getNumericAttribute().add(attribute);

        StringAttribute hostStringAttribute = new StringAttribute();
        hostStringAttribute.setAlias("host");
        hostStringAttribute.setValueExp("labels[host]");
        flinkTask.getStringAttribute().add(hostStringAttribute);

        // Define the resource type
        ResourceType resourceType = createStandardResourceType("flinkTaskManager");
        ResourceTypeMapper.getInstance().setResourceTypeMapper((rt) -> resourceType);

        // Collect!
        CollectionSet collectionSet = collectNephronMetrics(collection, flinkTask);

        // Verify
        List<String> collectionSetKeys = CollectionSetUtils.flatten(collectionSet);
        assertThat(collectionSetKeys, hasSize(26));
        assertThat(collectionSetKeys, hasItem("0/flinkTaskManager/42bb6e71af3820977305d89635f47dbf/flink-task-manager/Netwo_TotalMemoSegm[null,3278.0]"));
        assertThat(collectionSetKeys, hasItem("0/flinkTaskManager/42bb6e71af3820977305d89635f47dbf/flink-task-manager/JV_GaCo_G1_Ol_Ge_Co[null,1.0]"));
        assertThat(collectionSetKeys, hasItem("0/flinkTaskManager/42bb6e71af3820977305d89635f47dbf/flink-task-manager/host[jw_dev_2,null]"));
    }

    @Test
    public void canCollectFlinkTaskMetrics() {
        // Define our configuration
        Collection collection = new Collection();
        Group flinkTask = new Group();
        flinkTask.setName("flink-task");
        flinkTask.setFilterExp("name matches 'flink_taskmanager_job_task_.*'");
        flinkTask.setGroupByExp("labels[task_id]");
        flinkTask.setResourceType("flinkTask");

        NumericAttribute attribute = new NumericAttribute();
        attribute.setAliasExp("name.substring('flink_taskmanager_job_task_'.length())");
        flinkTask.getNumericAttribute().add(attribute);

        StringAttribute taskNameStringAttribute = new StringAttribute();
        taskNameStringAttribute.setAlias("task_name");
        taskNameStringAttribute.setValueExp("labels[task_name]");
        flinkTask.getStringAttribute().add(taskNameStringAttribute);

        StringAttribute jobIdAttribute = new StringAttribute();
        jobIdAttribute.setAlias("job_id");
        jobIdAttribute.setValueExp("labels[job_id]");
        flinkTask.getStringAttribute().add(jobIdAttribute);

        // Define the resource type
        ResourceType resourceType = createStandardResourceType("flinkTask");
        ResourceTypeMapper.getInstance().setResourceTypeMapper((rt) -> resourceType);

        // Collect!
        CollectionSet collectionSet = collectNephronMetrics(collection, flinkTask);

        // Verify
        List<String> collectionSetKeys = CollectionSetUtils.flatten(collectionSet);
        assertThat(collectionSetKeys, hasSize(629));
        assertThat(collectionSetKeys, hasItem("0/flinkTask/058ef95d84f46825bd5592f3e3fc122a/flink-task/numRecorOutPerSecon[null,0.1]"));
        assertThat(collectionSetKeys, hasItem("0/flinkTask/058ef95d84f46825bd5592f3e3fc122a/flink-task/task_name[FlowAnalyzer_CalculateFlowStatistics_CalculateTopHostsByExporterAndInterface_CalculateTopHostsByExporterAndInterface_top_k_per_key____FlowAnalyzer_CalculateFlowStatistics_CalculateTopHostsByExporterAndInterface_CalculateTopHostsByExporterAndInterface_flatten_Values_Map_ParMultiDo_Anonymous_____FlowAnalyzer_CalculateFlowStatistics_CalculateTopHostsByExporterAndInterface_CalculateTopHostsByExporterAndInterface_top_k_for_window_ParMultiDo_Anonymous_,null]"));
    }

    @Test
    public void canGatherLoadAverageMetrics() {
        // Define our configuration
        Collection collection = new Collection();
        Group nodeExporterLoadAverage = new Group();
        nodeExporterLoadAverage.setName("node-exporter-loadavg");
        nodeExporterLoadAverage.setFilterExp("name matches 'node_load.*'");
        nodeExporterLoadAverage.setResourceType("node");

        NumericAttribute attribute = new NumericAttribute();
        attribute.setAliasExp("name.substring('node_'.length())");
        nodeExporterLoadAverage.getNumericAttribute().add(attribute);

        // Collect!
        CollectionSet collectionSet = collect(collection, Lists.newArrayList(nodeExporterLoadAverage));

        // Verify
        List<String> collectionSetKeys = CollectionSetUtils.flatten(collectionSet);
        assertEquals(Arrays.asList("0/node-exporter-loadavg/load1[null,0.58]",
                "0/node-exporter-loadavg/load15[null,0.64]",
                "0/node-exporter-loadavg/load5[null,0.36]"), collectionSetKeys);
    }

    @Test
    public void canGatherFileSystemMetrics() {
        // Define our configuration
        Collection collection = new Collection();
        Group nodeExporterCpu = new Group();
        nodeExporterCpu.setName("node-exporter-filesystems");
        nodeExporterCpu.setFilterExp("name matches 'node_filesystem_.*' and labels[mountpoint] matches '.*home'");
        nodeExporterCpu.setGroupByExp("labels[mountpoint]");
        nodeExporterCpu.setResourceType("nodeExporterFilesytem");

        NumericAttribute attribute = new NumericAttribute();
        attribute.setAliasExp("name.substring('node_filesystem_'.length())");
        nodeExporterCpu.getNumericAttribute().add(attribute);

        StringAttribute fsTypeStringAttribute = new StringAttribute();
        fsTypeStringAttribute.setAlias("fstype");
        fsTypeStringAttribute.setValueExp("labels[fstype]");
        nodeExporterCpu.getStringAttribute().add(fsTypeStringAttribute);

        StringAttribute deviceStringAttribute = new StringAttribute();
        deviceStringAttribute.setAlias("device");
        deviceStringAttribute.setValueExp("labels[device]");
        nodeExporterCpu.getStringAttribute().add(deviceStringAttribute);

        // Define the resource type
        ResourceType resourceType = createStandardResourceType("nodeExporterFilesytem");
        ResourceTypeMapper.getInstance().setResourceTypeMapper((rt) -> resourceType);

        // Collect!
        CollectionSet collectionSet = collect(collection, Lists.newArrayList(nodeExporterCpu));

        // Verify
        List<String> collectionSetKeys = CollectionSetUtils.flatten(collectionSet);
        assertThat(collectionSetKeys, hasSize(8));
        assertThat(collectionSetKeys, hasItem("0/nodeExporterFilesytem/_rootfs_home/node-exporter-filesystems/files[null,2.4420352E7]"));
        assertThat(collectionSetKeys, hasItem("0/nodeExporterFilesytem/_rootfs_home/node-exporter-filesystems/device[/dev/mapper/optical-home,null]"));
    }

    @Test
    public void canGatherDiskMetrics() {
        // Define our configuration
        Collection collection = new Collection();
        Group nodeExporterDisks = new Group();
        nodeExporterDisks.setName("node-exporter-disks");
        nodeExporterDisks.setFilterExp("name matches 'node_disk_.*'");
        nodeExporterDisks.setGroupByExp("labels[device]");
        nodeExporterDisks.setResourceType("nodeExporterDisk");

        NumericAttribute attribute = new NumericAttribute();
        attribute.setAliasExp("name.substring('node_disk_'.length())");
        nodeExporterDisks.getNumericAttribute().add(attribute);

        // Define the resource type
        ResourceType resourceType = createStandardResourceType("nodeExporterDisk");
        ResourceTypeMapper.getInstance().setResourceTypeMapper((rt) -> resourceType);
        
        // Collect!
        CollectionSet collectionSet = collect(collection, Lists.newArrayList(nodeExporterDisks));

        // Verify
        List<String> collectionSetKeys = CollectionSetUtils.flatten(collectionSet);
        assertEquals(Arrays.asList("0/nodeExporterDisk/dm-0/node-exporter-disks/bytes_read[null,2539520.0]",
                "0/nodeExporterDisk/dm-0/node-exporter-disks/bytes_written[null,1318912.0]",
                "0/nodeExporterDisk/dm-0/node-exporter-disks/io_now[null,0.0]",
                "0/nodeExporterDisk/dm-0/node-exporter-disks/io_time_ms[null,604.0]",
                "0/nodeExporterDisk/dm-0/node-exporter-disks/io_time_weighted[null,755.0]",
                "0/nodeExporterDisk/dm-0/node-exporter-disks/read_time_ms[null,591.0]",
                "0/nodeExporterDisk/dm-0/node-exporter-disks/reads_completed[null,156.0]",
                "0/nodeExporterDisk/dm-0/node-exporter-disks/reads_merged[null,0.0]",
                "0/nodeExporterDisk/dm-0/node-exporter-disks/sectors_read[null,4960.0]",
                "0/nodeExporterDisk/dm-0/node-exporter-disks/sectors_written[null,2576.0]",
                "0/nodeExporterDisk/dm-0/node-exporter-disks/write_time_ms[null,164.0]",
                "0/nodeExporterDisk/dm-0/node-exporter-disks/writes_completed[null,322.0]",
                "0/nodeExporterDisk/dm-0/node-exporter-disks/writes_merged[null,0.0]",
                "0/nodeExporterDisk/dm-1/node-exporter-disks/bytes_read[null,1.792955392E9]",
                "0/nodeExporterDisk/dm-1/node-exporter-disks/bytes_written[null,8.90744832E8]",
                "0/nodeExporterDisk/dm-1/node-exporter-disks/io_now[null,2.0]",
                "0/nodeExporterDisk/dm-1/node-exporter-disks/io_time_ms[null,106110.0]",
                "0/nodeExporterDisk/dm-1/node-exporter-disks/io_time_weighted[null,1761109.0]",
                "0/nodeExporterDisk/dm-1/node-exporter-disks/read_time_ms[null,157601.0]",
                "0/nodeExporterDisk/dm-1/node-exporter-disks/reads_completed[null,76622.0]",
                "0/nodeExporterDisk/dm-1/node-exporter-disks/reads_merged[null,0.0]",
                "0/nodeExporterDisk/dm-1/node-exporter-disks/sectors_read[null,3501866.0]",
                "0/nodeExporterDisk/dm-1/node-exporter-disks/sectors_written[null,1739736.0]",
                "0/nodeExporterDisk/dm-1/node-exporter-disks/write_time_ms[null,1602698.0]",
                "0/nodeExporterDisk/dm-1/node-exporter-disks/writes_completed[null,95298.0]",
                "0/nodeExporterDisk/dm-1/node-exporter-disks/writes_merged[null,0.0]",
                "0/nodeExporterDisk/dm-2/node-exporter-disks/bytes_read[null,4.133188608E9]",
                "0/nodeExporterDisk/dm-2/node-exporter-disks/bytes_written[null,6.6522103808E10]",
                "0/nodeExporterDisk/dm-2/node-exporter-disks/io_now[null,0.0]",
                "0/nodeExporterDisk/dm-2/node-exporter-disks/io_time_ms[null,371775.0]",
                "0/nodeExporterDisk/dm-2/node-exporter-disks/io_time_weighted[null,7.0852048E7]",
                "0/nodeExporterDisk/dm-2/node-exporter-disks/read_time_ms[null,117412.0]",
                "0/nodeExporterDisk/dm-2/node-exporter-disks/reads_completed[null,317168.0]",
                "0/nodeExporterDisk/dm-2/node-exporter-disks/reads_merged[null,0.0]",
                "0/nodeExporterDisk/dm-2/node-exporter-disks/sectors_read[null,8072634.0]",
                "0/nodeExporterDisk/dm-2/node-exporter-disks/sectors_written[null,1.29925984E8]",
                "0/nodeExporterDisk/dm-2/node-exporter-disks/write_time_ms[null,7.0724706E7]",
                "0/nodeExporterDisk/dm-2/node-exporter-disks/writes_completed[null,2246867.0]",
                "0/nodeExporterDisk/dm-2/node-exporter-disks/writes_merged[null,0.0]",
                "0/nodeExporterDisk/sda/node-exporter-disks/bytes_read[null,1.809073664E9]",
                "0/nodeExporterDisk/sda/node-exporter-disks/bytes_written[null,8.92125696E8]",
                "0/nodeExporterDisk/sda/node-exporter-disks/io_now[null,0.0]",
                "0/nodeExporterDisk/sda/node-exporter-disks/io_time_ms[null,103955.0]",
                "0/nodeExporterDisk/sda/node-exporter-disks/io_time_weighted[null,711198.0]",
                "0/nodeExporterDisk/sda/node-exporter-disks/read_time_ms[null,109492.0]",
                "0/nodeExporterDisk/sda/node-exporter-disks/reads_completed[null,74745.0]",
                "0/nodeExporterDisk/sda/node-exporter-disks/reads_merged[null,2634.0]",
                "0/nodeExporterDisk/sda/node-exporter-disks/sectors_read[null,3533347.0]",
                "0/nodeExporterDisk/sda/node-exporter-disks/sectors_written[null,1742433.0]",
                "0/nodeExporterDisk/sda/node-exporter-disks/write_time_ms[null,600041.0]",
                "0/nodeExporterDisk/sda/node-exporter-disks/writes_completed[null,68129.0]",
                "0/nodeExporterDisk/sda/node-exporter-disks/writes_merged[null,32275.0]",
                "0/nodeExporterDisk/sdb/node-exporter-disks/bytes_read[null,4.135788032E9]",
                "0/nodeExporterDisk/sdb/node-exporter-disks/bytes_written[null,6.6522103808E10]",
                "0/nodeExporterDisk/sdb/node-exporter-disks/io_now[null,0.0]",
                "0/nodeExporterDisk/sdb/node-exporter-disks/io_time_ms[null,351586.0]",
                "0/nodeExporterDisk/sdb/node-exporter-disks/io_time_weighted[null,1.4474691E7]",
                "0/nodeExporterDisk/sdb/node-exporter-disks/read_time_ms[null,83011.0]",
                "0/nodeExporterDisk/sdb/node-exporter-disks/reads_completed[null,303217.0]",
                "0/nodeExporterDisk/sdb/node-exporter-disks/reads_merged[null,13779.0]",
                "0/nodeExporterDisk/sdb/node-exporter-disks/sectors_read[null,8077711.0]",
                "0/nodeExporterDisk/sdb/node-exporter-disks/sectors_written[null,1.29925984E8]",
                "0/nodeExporterDisk/sdb/node-exporter-disks/write_time_ms[null,1.4379594E7]",
                "0/nodeExporterDisk/sdb/node-exporter-disks/writes_completed[null,1024681.0]",
                "0/nodeExporterDisk/sdb/node-exporter-disks/writes_merged[null,1234629.0]"), collectionSetKeys);
    }

    @Test
    public void canGatherCpuMetrics() {
        // Define our configuration
        Collection collection = new Collection();
        Group nodeExporterCpu = new Group();
        nodeExporterCpu.setName("node-exporter-cpu");
        nodeExporterCpu.setFilterExp("name matches 'node_cpu'");
        nodeExporterCpu.setGroupByExp("labels[cpu]");
        nodeExporterCpu.setResourceType("nodeExporterCPU");

        NumericAttribute attribute = new NumericAttribute();
        attribute.setAliasExp("labels[mode]");
        nodeExporterCpu.getNumericAttribute().add(attribute);

        // Define the resource type
        ResourceType resourceType = createStandardResourceType("nodeExporterCPU");
        ResourceTypeMapper.getInstance().setResourceTypeMapper((rt) -> resourceType);
        
        // Collect!
        CollectionSet collectionSet = collect(collection, Lists.newArrayList(nodeExporterCpu));

        // Verify
        List<String> collectionSetKeys = CollectionSetUtils.flatten(collectionSet);
        assertEquals(Arrays.asList("0/nodeExporterCPU/cpu0/node-exporter-cpu/guest[null,0.0]",
                "0/nodeExporterCPU/cpu0/node-exporter-cpu/idle[null,16594.88]",
                "0/nodeExporterCPU/cpu0/node-exporter-cpu/iowait[null,163.99]",
                "0/nodeExporterCPU/cpu0/node-exporter-cpu/irq[null,62.55]",
                "0/nodeExporterCPU/cpu0/node-exporter-cpu/nice[null,1134.72]",
                "0/nodeExporterCPU/cpu0/node-exporter-cpu/softirq[null,58.2]",
                "0/nodeExporterCPU/cpu0/node-exporter-cpu/steal[null,0.0]",
                "0/nodeExporterCPU/cpu0/node-exporter-cpu/system[null,316.98]",
                "0/nodeExporterCPU/cpu0/node-exporter-cpu/user[null,1638.27]",
                "0/nodeExporterCPU/cpu1/node-exporter-cpu/guest[null,0.0]",
                "0/nodeExporterCPU/cpu1/node-exporter-cpu/idle[null,17790.51]",
                "0/nodeExporterCPU/cpu1/node-exporter-cpu/iowait[null,71.6]",
                "0/nodeExporterCPU/cpu1/node-exporter-cpu/irq[null,39.27]",
                "0/nodeExporterCPU/cpu1/node-exporter-cpu/nice[null,660.0]",
                "0/nodeExporterCPU/cpu1/node-exporter-cpu/softirq[null,34.22]",
                "0/nodeExporterCPU/cpu1/node-exporter-cpu/steal[null,0.0]",
                "0/nodeExporterCPU/cpu1/node-exporter-cpu/system[null,207.24]",
                "0/nodeExporterCPU/cpu1/node-exporter-cpu/user[null,1182.24]",
                "0/nodeExporterCPU/cpu2/node-exporter-cpu/guest[null,0.0]",
                "0/nodeExporterCPU/cpu2/node-exporter-cpu/idle[null,16907.35]",
                "0/nodeExporterCPU/cpu2/node-exporter-cpu/iowait[null,134.94]",
                "0/nodeExporterCPU/cpu2/node-exporter-cpu/irq[null,35.18]",
                "0/nodeExporterCPU/cpu2/node-exporter-cpu/nice[null,928.18]",
                "0/nodeExporterCPU/cpu2/node-exporter-cpu/softirq[null,23.12]",
                "0/nodeExporterCPU/cpu2/node-exporter-cpu/steal[null,0.0]",
                "0/nodeExporterCPU/cpu2/node-exporter-cpu/system[null,316.74]",
                "0/nodeExporterCPU/cpu2/node-exporter-cpu/user[null,1616.54]",
                "0/nodeExporterCPU/cpu3/node-exporter-cpu/guest[null,0.0]",
                "0/nodeExporterCPU/cpu3/node-exporter-cpu/idle[null,17780.12]",
                "0/nodeExporterCPU/cpu3/node-exporter-cpu/iowait[null,52.55]",
                "0/nodeExporterCPU/cpu3/node-exporter-cpu/irq[null,26.66]",
                "0/nodeExporterCPU/cpu3/node-exporter-cpu/nice[null,696.89]",
                "0/nodeExporterCPU/cpu3/node-exporter-cpu/softirq[null,9.73]",
                "0/nodeExporterCPU/cpu3/node-exporter-cpu/steal[null,0.0]",
                "0/nodeExporterCPU/cpu3/node-exporter-cpu/system[null,203.84]",
                "0/nodeExporterCPU/cpu3/node-exporter-cpu/user[null,1217.81]",
                "0/nodeExporterCPU/cpu4/node-exporter-cpu/guest[null,0.0]",
                "0/nodeExporterCPU/cpu4/node-exporter-cpu/idle[null,16817.38]",
                "0/nodeExporterCPU/cpu4/node-exporter-cpu/iowait[null,121.42]",
                "0/nodeExporterCPU/cpu4/node-exporter-cpu/irq[null,29.09]",
                "0/nodeExporterCPU/cpu4/node-exporter-cpu/nice[null,986.56]",
                "0/nodeExporterCPU/cpu4/node-exporter-cpu/softirq[null,12.99]",
                "0/nodeExporterCPU/cpu4/node-exporter-cpu/steal[null,0.0]",
                "0/nodeExporterCPU/cpu4/node-exporter-cpu/system[null,311.9]",
                "0/nodeExporterCPU/cpu4/node-exporter-cpu/user[null,1696.0]",
                "0/nodeExporterCPU/cpu5/node-exporter-cpu/guest[null,0.0]",
                "0/nodeExporterCPU/cpu5/node-exporter-cpu/idle[null,17891.54]",
                "0/nodeExporterCPU/cpu5/node-exporter-cpu/iowait[null,44.88]",
                "0/nodeExporterCPU/cpu5/node-exporter-cpu/irq[null,20.32]",
                "0/nodeExporterCPU/cpu5/node-exporter-cpu/nice[null,570.46]",
                "0/nodeExporterCPU/cpu5/node-exporter-cpu/softirq[null,7.6]",
                "0/nodeExporterCPU/cpu5/node-exporter-cpu/steal[null,0.0]",
                "0/nodeExporterCPU/cpu5/node-exporter-cpu/system[null,202.13]",
                "0/nodeExporterCPU/cpu5/node-exporter-cpu/user[null,1252.44]",
                "0/nodeExporterCPU/cpu6/node-exporter-cpu/guest[null,0.0]",
                "0/nodeExporterCPU/cpu6/node-exporter-cpu/idle[null,16782.14]",
                "0/nodeExporterCPU/cpu6/node-exporter-cpu/iowait[null,146.53]",
                "0/nodeExporterCPU/cpu6/node-exporter-cpu/irq[null,26.75]",
                "0/nodeExporterCPU/cpu6/node-exporter-cpu/nice[null,1001.04]",
                "0/nodeExporterCPU/cpu6/node-exporter-cpu/softirq[null,12.32]",
                "0/nodeExporterCPU/cpu6/node-exporter-cpu/steal[null,0.0]",
                "0/nodeExporterCPU/cpu6/node-exporter-cpu/system[null,300.31]",
                "0/nodeExporterCPU/cpu6/node-exporter-cpu/user[null,1705.1]",
                "0/nodeExporterCPU/cpu7/node-exporter-cpu/guest[null,0.0]",
                "0/nodeExporterCPU/cpu7/node-exporter-cpu/idle[null,17955.02]",
                "0/nodeExporterCPU/cpu7/node-exporter-cpu/iowait[null,49.39]",
                "0/nodeExporterCPU/cpu7/node-exporter-cpu/irq[null,17.68]",
                "0/nodeExporterCPU/cpu7/node-exporter-cpu/nice[null,553.09]",
                "0/nodeExporterCPU/cpu7/node-exporter-cpu/softirq[null,7.85]",
                "0/nodeExporterCPU/cpu7/node-exporter-cpu/steal[null,0.0]",
                "0/nodeExporterCPU/cpu7/node-exporter-cpu/system[null,200.86]",
                "0/nodeExporterCPU/cpu7/node-exporter-cpu/user[null,1203.71]"), collectionSetKeys);
    }

    private CollectionSet collect(Collection collection, List<Group> groups) {
        return collect(collection, groups, "metrics");
    }

    private CollectionSet collectNephronMetrics(Collection collection, Group... groups) {
        return collect(collection, Arrays.asList(groups), "nephron_on_flink");
    }

    private CollectionSet collect(Collection collection, List<Group> groups, String path) {
        // Create the agent
        OnmsNode node = mock(OnmsNode.class);
        OnmsIpInterface iface = mock(OnmsIpInterface.class);
        when(iface.getNode()).thenReturn(node);
        when(iface.getIpAddress()).thenReturn(InetAddressUtils.getLocalHostAddress());

        IpInterfaceDao ifaceDao = mock(IpInterfaceDao.class);
        when(ifaceDao.load(1)).thenReturn(iface);
        PlatformTransactionManager transMgr = mock(PlatformTransactionManager.class);
        final CollectionAgent agent = DefaultCollectionAgent.create(1, ifaceDao, transMgr);

        PrometheusDataCollectionConfigDao collectionDao = mock(PrometheusDataCollectionConfigDao.class);
        when(collectionDao.getCollectionByName(any())).thenReturn(collection);
        when(collectionDao.getGroupsForCollection(collection)).thenReturn(groups);
        collector.setPrometheusCollectionDao(collectionDao);

        try {
            Map<String, Object> serviceParams = new ImmutableMap.Builder<String, Object>()
                    .put("collection", "default")
                    .put("url", String.format("http://127.0.0.1:%d/" + path, wireMockRule.port()))
                    .build();
            Map<String, Object> runtimeParams = collector.getRuntimeAttributes(agent, serviceParams);
            Map<String, Object> allParams = new HashMap<>();
            allParams.putAll(serviceParams);
            allParams.putAll(runtimeParams);

            return collector.collect(agent, allParams);
        } catch (CollectionException e) {
            throw new RuntimeException(e);
        }
    }

    private static ResourceType createStandardResourceType(String name) {
        final ResourceType resourceType = new ResourceType();
        resourceType.setName(name);
        resourceType.setLabel("Label for: " + name);
        resourceType.setResourceLabel("${instance}");
        final StorageStrategy storageStrategy = new StorageStrategy();
        storageStrategy.setClazz(IndexStorageStrategy.class.getCanonicalName());
        resourceType.setStorageStrategy(storageStrategy);
        final PersistenceSelectorStrategy persistenceSelectorStrategy = new PersistenceSelectorStrategy();
        persistenceSelectorStrategy.setClazz(PersistAllSelectorStrategy.class.getCanonicalName());
        resourceType.setPersistenceSelectorStrategy(persistenceSelectorStrategy);
        return resourceType;
    }

}
