/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.core.ipc.sink.offheap;

import com.swrve.ratelimitedlogger.RateLimitedLog;
import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import org.opennms.core.ipc.sink.api.DispatchQueue;
import org.opennms.core.ipc.sink.api.QueueCreateFailedException;
import org.opennms.core.ipc.sink.api.ReadFailedException;
import org.opennms.core.ipc.sink.api.WriteFailedException;
import org.opennms.core.ipc.sink.offheap.DataBlock;
import org.opennms.core.ipc.sink.offheap.MemoryDataBlock;
import org.opennms.core.ipc.sink.offheap.RocksDBOffHeapDataBlock;
import org.rocksdb.Cache;
import org.rocksdb.CompressionType;
import org.rocksdb.Env;
import org.rocksdb.FlushOptions;
import org.rocksdb.LRUCache;
import org.rocksdb.Options;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.rocksdb.SstFileManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataBlocksOffHeapQueue<T>
implements DispatchQueue<T> {
    private static final Logger LOG = LoggerFactory.getLogger(DataBlocksOffHeapQueue.class);
    private static final RateLimitedLog RATE_LIMITED_LOGGER = RateLimitedLog.withRateLimit((Logger)LOG).maxRate(5).every(Duration.ofSeconds(30L)).build();
    private final Lock headLock = new ReentrantLock(true);
    private final Lock tailLock = new ReentrantLock(true);
    private final Function<T, byte[]> serializer;
    private final Function<byte[], T> deserializer;
    private final int batchSize;
    private final BlockingQueue<DataBlock<T>> mainQueue;
    private DataBlock<T> tailBlock;
    private AtomicInteger memoryBlockCount = new AtomicInteger(0);
    private AtomicInteger getOffHeapBlockCount = new AtomicInteger(0);
    private int inMemoryQueueSize;
    private long maxOffHeapFileSize;
    private RocksDB rocksdb;
    private final SstFileManager sstFileManager;
    private final Options options;
    private final LRUCache cache;

    public DataBlocksOffHeapQueue(Function<T, byte[]> serializer, Function<byte[], T> deserializer, String moduleName, Path filePath, int inMemoryQueueSize, int batchSize, long maxOffHeapFileSize, long writeBufferSize, long cacheSize) throws QueueCreateFailedException {
        if (inMemoryQueueSize < 1) {
            throw new IllegalArgumentException("In memory queue size must be greater than 0");
        }
        if (inMemoryQueueSize % batchSize != 0) {
            throw new IllegalArgumentException("In memory queue size must be a multiple of batch size");
        }
        if (maxOffHeapFileSize < 0L) {
            throw new IllegalArgumentException("Max offheap file size must be either 0 or a positive integer");
        }
        if (cacheSize < 0L) {
            throw new IllegalArgumentException("Cache size must be either 0 or a positive integer");
        }
        this.mainQueue = new LinkedBlockingQueue<DataBlock<T>>();
        this.serializer = Objects.requireNonNull(serializer);
        this.deserializer = Objects.requireNonNull(deserializer);
        Objects.requireNonNull(moduleName);
        this.batchSize = batchSize;
        this.inMemoryQueueSize = inMemoryQueueSize;
        this.maxOffHeapFileSize = maxOffHeapFileSize;
        File file = Paths.get(filePath.toString(), moduleName).toFile();
        if (!file.isDirectory() && !file.mkdirs()) {
            throw new QueueCreateFailedException("Fail make dir. " + file);
        }
        this.options = new Options();
        this.options.setCreateIfMissing(true);
        this.options.setCompressionType(CompressionType.SNAPPY_COMPRESSION);
        this.options.setEnableBlobFiles(true);
        this.options.setEnableBlobGarbageCollection(true);
        this.options.setMinBlobSize(100000L);
        this.options.setBlobCompressionType(CompressionType.SNAPPY_COMPRESSION);
        this.options.setWriteBufferSize(writeBufferSize);
        this.options.setBlobFileSize(this.options.writeBufferSize());
        this.options.setTargetFileSizeBase(0x4000000L);
        this.options.setMaxBytesForLevelBase(this.options.targetFileSizeBase() * 10L);
        this.options.setMaxBackgroundJobs(Math.max(Runtime.getRuntime().availableProcessors(), 3));
        this.cache = new LRUCache(cacheSize, 8);
        this.options.setRowCache((Cache)this.cache);
        try {
            Env env = Env.getDefault();
            this.sstFileManager = new SstFileManager(env);
            this.options.setSstFileManager(this.sstFileManager);
            this.rocksdb = RocksDB.open((Options)this.options, (String)file.getAbsolutePath());
            this.restoreDataFromOffHeap();
            if (this.mainQueue.isEmpty()) {
                this.createDataBlock(true);
            }
        }
        catch (RocksDBException ex) {
            throw new QueueCreateFailedException((Throwable)ex);
        }
    }

    public DataBlocksOffHeapQueue(Function<T, byte[]> serializer, Function<byte[], T> deserializer, String moduleName, Path filePath, int inMemoryQueueSize, int batchSize, long maxOffHeapFileSize) throws QueueCreateFailedException {
        this(serializer, deserializer, moduleName, filePath, inMemoryQueueSize, batchSize, maxOffHeapFileSize, 0x6400000L, 0x20000000L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void restoreDataFromOffHeap() throws RocksDBException {
        long keyCount = this.rocksdb.getLongProperty("rocksdb.estimate-num-keys");
        LOG.info("Current keys in db: {}", (Object)keyCount);
        if (keyCount <= 0L) {
            return;
        }
        try (RocksIterator ite = this.rocksdb.newIterator();){
            RocksDBOffHeapDataBlock<T> lastBlock = null;
            ite.seekToFirst();
            while (ite.isValid()) {
                String key = new String(ite.key());
                RocksDBOffHeapDataBlock<T> newBlock = new RocksDBOffHeapDataBlock<T>(key, this.batchSize, this.serializer, this.deserializer, this.rocksdb, ite.value());
                if (lastBlock != null) {
                    lastBlock.setNextDataBlock(newBlock);
                }
                this.mainQueue.add(newBlock);
                this.getOffHeapBlockCount.incrementAndGet();
                lastBlock = newBlock;
                ite.next();
            }
        }
    }

    private void createDataBlock(boolean force) {
        DataBlock newBlock;
        if (!force && this.tailBlock != null && this.tailBlock.size() < this.batchSize) {
            return;
        }
        if (this.tailBlock == null || !this.isMemoryFull()) {
            newBlock = new MemoryDataBlock(this.batchSize);
            this.memoryBlockCount.incrementAndGet();
        } else {
            newBlock = new RocksDBOffHeapDataBlock<T>(this.batchSize, this.serializer, this.deserializer, this.rocksdb);
            this.getOffHeapBlockCount.incrementAndGet();
        }
        RATE_LIMITED_LOGGER.debug("Create {}, mem: {}, disk: {}", new Object[]{newBlock, this.memoryBlockCount.get(), this.getOffHeapBlockCount.get()});
        if (this.tailBlock != null) {
            this.tailBlock.setNextDataBlock(newBlock);
        }
        this.tailBlock = newBlock;
        this.mainQueue.add(newBlock);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DispatchQueue.EnqueueResult enqueue(T message, String key) throws WriteFailedException {
        try {
            DataBlock<T> dataBlock = this.tailBlock;
            synchronized (dataBlock) {
                while (this.isFull()) {
                    this.tailBlock.wait(10L);
                }
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        this.tailLock.lock();
        boolean status = this.tailBlock.enqueue(key, message);
        if (!status) {
            this.tailLock.unlock();
            return DispatchQueue.EnqueueResult.DEFERRED;
        }
        this.createDataBlock(false);
        this.tailLock.unlock();
        return DispatchQueue.EnqueueResult.IMMEDIATE;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Map.Entry<String, T> dequeue() throws InterruptedException {
        this.headLock.lock();
        Map.Entry<String, T> data = null;
        try {
            while (data == null) {
                data = this.readData();
            }
        }
        catch (ReadFailedException e) {
            LOG.error("Fail to dequeue. {}", (Object)e.getMessage());
            Map.Entry<String, T> entry = null;
            return entry;
        }
        finally {
            this.headLock.unlock();
        }
        return data;
    }

    private Map.Entry<String, T> readData() throws ReadFailedException, InterruptedException {
        DataBlock head;
        Map.Entry data = null;
        boolean headTailEqual = false;
        if (this.mainQueue.peek() == this.tailBlock) {
            if (!this.tailLock.tryLock(1L, TimeUnit.MILLISECONDS)) {
                return null;
            }
            headTailEqual = true;
        }
        if ((head = (DataBlock)this.mainQueue.peek()).size() > 0) {
            data = head.dequeue();
            if (headTailEqual) {
                this.tailLock.unlock();
            } else {
                this.removeHeadBlockIfNeeded();
            }
        } else if (headTailEqual) {
            this.tailLock.unlock();
        }
        return data;
    }

    private void removeHeadBlockIfNeeded() throws ReadFailedException {
        DataBlock head = (DataBlock)this.mainQueue.peek();
        if (head.size() <= 0 && this.mainQueue.size() > 1) {
            DataBlock tmpHead = (DataBlock)this.mainQueue.remove();
            if (tmpHead != head) {
                LOG.error("Queue is modified. May have data lost");
            }
            if (head instanceof MemoryDataBlock) {
                this.memoryBlockCount.decrementAndGet();
            } else {
                this.getOffHeapBlockCount.decrementAndGet();
            }
            ((DataBlock)this.mainQueue.peek()).notifyNextDataBlock();
        }
    }

    private boolean isOffHeapFull() {
        return this.sstFileManager.getTotalSize() >= this.maxOffHeapFileSize;
    }

    private boolean isMemoryFull() {
        MemoryDataBlock lastBlock = this.tailBlock instanceof MemoryDataBlock ? (MemoryDataBlock)this.tailBlock : null;
        int remain = this.inMemoryQueueSize - (this.memoryBlockCount.get() - (lastBlock == null ? 0 : 1)) * this.batchSize;
        if (remain > 0 && lastBlock == null) {
            return false;
        }
        if (lastBlock != null) {
            return remain - lastBlock.size() <= 0;
        }
        return true;
    }

    public boolean isFull() {
        if (this.isMemoryFull()) {
            return this.isOffHeapFull();
        }
        return false;
    }

    public int getSize() {
        return this.mainQueue.stream().map(DataBlock::size).reduce(Integer::sum).orElse(0);
    }

    public int getMemoryBlockCount() {
        return this.memoryBlockCount.get();
    }

    public int getOffHeapBlockCount() {
        return this.getOffHeapBlockCount.get();
    }

    public long getOffHeapFileSize() {
        return this.sstFileManager.getTotalSize();
    }

    public void flushOffHeap() throws RocksDBException {
        try (FlushOptions fOptions = new FlushOptions();){
            this.rocksdb.flush(fOptions.setWaitForFlush(true).setAllowWriteStall(true));
        }
    }

    public void shutdown() {
        this.rocksdb.close();
        this.sstFileManager.close();
        this.options.close();
        this.cache.close();
    }
}

