/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.features.alarms.history.elastic;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.cache.CacheLoader;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.gson.Gson;
import io.searchbox.action.Action;
import io.searchbox.action.BulkableAction;
import io.searchbox.core.Bulk;
import io.searchbox.core.Index;
import io.searchbox.core.Search;
import io.searchbox.core.SearchResult;
import io.searchbox.core.search.aggregation.TopHitsAggregation;
import java.io.IOException;
import java.time.Instant;
import java.time.temporal.TemporalAccessor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.validation.constraints.NotNull;
import org.opennms.core.cache.Cache;
import org.opennms.core.cache.CacheBuilder;
import org.opennms.core.cache.CacheConfig;
import org.opennms.core.time.PseudoClock;
import org.opennms.features.alarms.history.elastic.CompositeAggregation;
import org.opennms.features.alarms.history.elastic.ElasticAlarmHistoryRepository;
import org.opennms.features.alarms.history.elastic.ElasticAlarmMetrics;
import org.opennms.features.alarms.history.elastic.PersistenceException;
import org.opennms.features.alarms.history.elastic.QueryProvider;
import org.opennms.features.alarms.history.elastic.TimeRange;
import org.opennms.features.alarms.history.elastic.dto.AlarmDocumentDTO;
import org.opennms.features.alarms.history.elastic.dto.AlarmDocumentFactory;
import org.opennms.features.alarms.history.elastic.dto.NodeDocumentDTO;
import org.opennms.features.alarms.history.elastic.mapping.MapStructDocumentImpl;
import org.opennms.features.alarms.history.elastic.tasks.BulkDeleteTask;
import org.opennms.features.alarms.history.elastic.tasks.IndexAlarmsTask;
import org.opennms.features.alarms.history.elastic.tasks.Task;
import org.opennms.features.alarms.history.elastic.tasks.TaskVisitor;
import org.opennms.features.jest.client.JestClientWithCircuitBreaker;
import org.opennms.features.jest.client.bulk.BulkException;
import org.opennms.features.jest.client.bulk.BulkRequest;
import org.opennms.features.jest.client.bulk.BulkWrapper;
import org.opennms.features.jest.client.index.IndexSelector;
import org.opennms.features.jest.client.index.IndexStrategy;
import org.opennms.features.jest.client.template.IndexSettings;
import org.opennms.features.jest.client.template.TemplateInitializer;
import org.opennms.netmgt.alarmd.api.AlarmCallbackStateTracker;
import org.opennms.netmgt.alarmd.api.AlarmLifecycleListener;
import org.opennms.netmgt.model.OnmsAlarm;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ElasticAlarmIndexer
implements AlarmLifecycleListener,
Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(ElasticAlarmIndexer.class);
    private static final Gson gson = new Gson();
    public static final int DEFAULT_TASK_QUEUE_CAPACITY = 5000;
    public static final String INDEX_NAME = "opennms-alarms";
    private final AlarmCallbackStateTracker stateTracker = new AlarmCallbackStateTracker();
    private final QueryProvider queryProvider = new QueryProvider();
    private final JestClientWithCircuitBreaker client;
    private final TemplateInitializer templateInitializer;
    private final LinkedBlockingDeque<Task> taskQueue;
    private final IndexStrategy indexStrategy;
    private final IndexSelector indexSelector;
    private int bulkRetryCount = 3;
    private int batchSize = 200;
    private boolean usePseudoClock = false;
    private boolean indexAllUpdates = false;
    private long alarmReindexDurationMs = TimeUnit.HOURS.toMillis(1L);
    private long lookbackPeriodMs = ElasticAlarmHistoryRepository.DEFAULT_LOOKBACK_PERIOD_MS;
    private final List<AlarmDocumentDTO> alarmDocumentsToIndex = new LinkedList<AlarmDocumentDTO>();
    private Map<Integer, AlarmDocumentDTO> alarmDocumentsById = new LinkedHashMap<Integer, AlarmDocumentDTO>();
    private final ExecutorService executor = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setNameFormat("ElasticAlarmIndexer").build());
    private final AtomicBoolean stopped = new AtomicBoolean(false);
    private Timer timer;
    private final Function<OnmsAlarm, AlarmDocumentDTO> documentMapper;
    private final AlarmDocumentFactory documentFactory;
    private final ElasticAlarmMetrics alarmsToESMetrics;
    private final IndexSettings indexSettings;

    public ElasticAlarmIndexer(MetricRegistry metrics, JestClientWithCircuitBreaker client, TemplateInitializer templateInitializer) {
        this(metrics, client, templateInitializer, new CacheConfig("nodes-for-alarms-in-es"), 5000, IndexStrategy.MONTHLY, new IndexSettings());
    }

    public ElasticAlarmIndexer(MetricRegistry metrics, JestClientWithCircuitBreaker client, TemplateInitializer templateInitializer, CacheConfig nodeCacheConfig, int taskQueueCapacity, IndexStrategy indexStrategy, IndexSettings indexSettings) {
        MapStructDocumentImpl documentImpl;
        this.client = Objects.requireNonNull(client);
        this.templateInitializer = Objects.requireNonNull(templateInitializer);
        Cache nodeInfoCache = new CacheBuilder().withConfig(nodeCacheConfig).withCacheLoader((CacheLoader)new CacheLoader<Integer, Optional<NodeDocumentDTO>>(){

            public Optional<NodeDocumentDTO> load(@NotNull Integer nodeId) {
                return Optional.empty();
            }
        }).build();
        this.documentMapper = documentImpl = new MapStructDocumentImpl((Cache<Integer, Optional<NodeDocumentDTO>>)nodeInfoCache, this::getCurrentTimeMillis);
        this.documentFactory = documentImpl;
        this.taskQueue = new LinkedBlockingDeque(taskQueueCapacity);
        this.alarmsToESMetrics = new ElasticAlarmMetrics(metrics, this.taskQueue);
        this.indexStrategy = Objects.requireNonNull(indexStrategy);
        this.indexSettings = Objects.requireNonNull(indexSettings);
        this.indexSelector = new IndexSelector(indexSettings, INDEX_NAME, indexStrategy, 0L);
    }

    public void init() {
        if (this.stopped.get()) {
            throw new IllegalStateException("Already destroyed.");
        }
        this.executor.execute(this);
        this.timer = new Timer("ElasticAlarmIndexer");
        this.timer.schedule(new TimerTask(){

            @Override
            public void run() {
                try {
                    ElasticAlarmIndexer.this.flushDocumentsToIndexToTaskQueue();
                }
                catch (Exception e) {
                    LOG.error("Flush failed.", (Throwable)e);
                }
            }
        }, 500L, 500L);
    }

    public void destroy() {
        this.stopped.set(true);
        this.timer.cancel();
        this.executor.shutdown();
    }

    @Override
    public void run() {
        final AtomicLong lastbulkDeleteWithNoChanges = new AtomicLong(-1L);
        this.templateInitializer.initialize();
        while (!this.stopped.get()) {
            try {
                Task task = this.taskQueue.take();
                task.visit(new TaskVisitor(){

                    @Override
                    public void indexAlarms(List<AlarmDocumentDTO> docs) {
                        LinkedHashMap<String, AlarmDocumentDTO> deduplicatedDocs = new LinkedHashMap<String, AlarmDocumentDTO>();
                        for (AlarmDocumentDTO doc : docs) {
                            deduplicatedDocs.put(String.format("%d-%s", doc.getId(), doc.getUpdateTime()), doc);
                        }
                        docs = new ArrayList(deduplicatedDocs.values());
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Indexing documents for alarms with ids: {}", docs.stream().map(AlarmDocumentDTO::getId).collect(Collectors.toList()));
                        }
                        try (Timer.Context ctx = ElasticAlarmIndexer.this.alarmsToESMetrics.getBulkIndexTimer().time();){
                            ElasticAlarmIndexer.this.bulkInsert(docs);
                            LOG.debug("Successfully indexed {} documents.", (Object)docs.size());
                            ElasticAlarmIndexer.this.alarmsToESMetrics.getBulkIndexSizeHistogram().update(docs.size());
                        }
                        catch (IOException | PersistenceException e) {
                            LOG.error("Indexing {} documents failed. These documents will be lost.", (Object)docs.size(), (Object)e);
                            ElasticAlarmIndexer.this.alarmsToESMetrics.getTasksFailedCounter().inc();
                        }
                    }

                    @Override
                    public void deleteAlarmsWithoutIdsIn(Set<Integer> alarmIdsToKeep, long time) {
                        long includeUpdatesAfter = Math.max(time - ElasticAlarmIndexer.this.lookbackPeriodMs, 0L);
                        if (lastbulkDeleteWithNoChanges.get() > 0L) {
                            includeUpdatesAfter = lastbulkDeleteWithNoChanges.get();
                        }
                        LOG.debug("Marking documents without ids in: {} as deleted for time: {}", alarmIdsToKeep, (Object)time);
                        Timer.Context ctx = ElasticAlarmIndexer.this.alarmsToESMetrics.getBulkDeleteTimer().time();
                        try {
                            LinkedList alarms = new LinkedList();
                            Integer afterAlarmWithId = null;
                            while (true) {
                                CompositeAggregation alarmsById;
                                SearchResult result;
                                TimeRange timeRange = new TimeRange(includeUpdatesAfter, time);
                                String query = ElasticAlarmIndexer.this.queryProvider.getActiveAlarmIdsAtTimeAndExclude(timeRange, alarmIdsToKeep, afterAlarmWithId);
                                Search.Builder search = new Search.Builder(query);
                                List indices = ElasticAlarmIndexer.this.indexSelector.getIndexNames(timeRange.getStart(), timeRange.getEnd());
                                search.addIndices((Collection)indices);
                                search.setParameter("ignore_unavailable", (Object)"true");
                                LOG.debug("Executing query on {}: {}", (Object)indices, (Object)query);
                                try {
                                    result = (SearchResult)ElasticAlarmIndexer.this.client.execute((Action)search.build());
                                }
                                catch (IOException e) {
                                    LOG.error("Querying for active alarms failed.", (Throwable)e);
                                    if (ctx != null) {
                                        ctx.close();
                                    }
                                    return;
                                }
                                if (!result.isSucceeded()) {
                                    LOG.error("Querying for active alarms failed with: {}", (Object)result.getErrorMessage());
                                }
                                if ((alarmsById = (CompositeAggregation)result.getAggregations().getAggregation("alarms_by_id", CompositeAggregation.class)) == null) break;
                                for (CompositeAggregation.Entry entry : alarmsById.getBuckets()) {
                                    TopHitsAggregation topHitsAggregation = entry.getTopHitsAggregation("latest_alarm");
                                    List hits = topHitsAggregation.getHits(AlarmDocumentDTO.class);
                                    hits.stream().map(h -> (AlarmDocumentDTO)h.source).forEach(alarms::add);
                                }
                                if (!alarmsById.hasAfterKey()) break;
                                afterAlarmWithId = alarmsById.getAfterKey().get("alarm_id").getAsInt();
                            }
                            if (!alarms.isEmpty()) {
                                List deletes = alarms.stream().map(a -> ElasticAlarmIndexer.this.documentFactory.createAlarmDocumentForDelete(a.getId(), a.getReductionKey())).collect(Collectors.toList());
                                if (LOG.isDebugEnabled()) {
                                    LOG.debug("Deleting alarms with IDs: {}", (Object)deletes.stream().map(a -> Integer.toString(a.getId())).collect(Collectors.joining(",")));
                                }
                                for (List partition : Lists.partition(deletes, (int)ElasticAlarmIndexer.this.batchSize)) {
                                    this.indexAlarms(partition);
                                }
                            } else {
                                LOG.debug("Did not find any extraneous alarms that need to be deleted.");
                                lastbulkDeleteWithNoChanges.set(time);
                            }
                        }
                        finally {
                            if (ctx != null) {
                                try {
                                    ctx.close();
                                }
                                catch (Throwable throwable) {
                                    Throwable throwable2;
                                    throwable2.addSuppressed(throwable);
                                }
                            }
                        }
                    }
                });
            }
            catch (InterruptedException e) {
                LOG.info("Interrupted. Stopping.");
                return;
            }
            catch (Exception e) {
                LOG.error("Handling of task failed.", (Throwable)e);
                this.alarmsToESMetrics.getTasksFailedCounter().inc();
            }
        }
    }

    public void bulkInsert(List<AlarmDocumentDTO> alarmDocuments) throws PersistenceException, IOException {
        BulkRequest bulkRequest = new BulkRequest(this.client, alarmDocuments, documents -> {
            Bulk.Builder bulkBuilder = new Bulk.Builder();
            for (AlarmDocumentDTO alarmDocument : alarmDocuments) {
                String index = this.indexStrategy.getIndex(this.indexSettings, INDEX_NAME, (TemporalAccessor)Instant.ofEpochMilli(alarmDocument.getUpdateTime()));
                Index.Builder indexBuilder = (Index.Builder)new Index.Builder((Object)alarmDocument).index(index);
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Adding index action on index: {} with payload: {}", (Object)index, (Object)gson.toJson((Object)alarmDocument));
                }
                bulkBuilder.addAction((BulkableAction)indexBuilder.build());
            }
            return new BulkWrapper(bulkBuilder);
        }, this.bulkRetryCount);
        try {
            bulkRequest.execute();
        }
        catch (BulkException ex) {
            List failedItems = ex.getBulkResult() != null ? ex.getBulkResult().getFailedItems() : Collections.emptyList();
            throw new PersistenceException(ex.getMessage(), failedItems);
        }
        catch (IOException ex) {
            LOG.error("An error occurred while executing the given request: {}", (Object)ex.getMessage(), (Object)ex);
            throw ex;
        }
    }

    public synchronized void preHandleAlarmSnapshot() {
        this.stateTracker.startTrackingAlarms();
    }

    public synchronized void handleAlarmSnapshot(List<OnmsAlarm> alarms) {
        LOG.debug("Got snapshot with {} alarms.", (Object)alarms.size());
        this.flushDocumentsToIndexToTaskQueue();
        List alarmDocuments = alarms.stream().filter(a -> !this.stateTracker.wasAlarmWithIdUpdated(a.getId().intValue())).map(this::getDocumentIfNeedsIndexing).flatMap(o -> o.map(Stream::of).orElseGet(Stream::empty)).collect(Collectors.toList());
        if (!alarmDocuments.isEmpty()) {
            for (List partition : Lists.partition(alarmDocuments, (int)this.batchSize)) {
                this.taskQueue.add(new IndexAlarmsTask(partition));
            }
        }
        HashSet<Integer> alarmIdsToKeep = new HashSet<Integer>(this.stateTracker.getUpdatedAlarmIds());
        alarms.stream().map(OnmsAlarm::getId).forEach(alarmIdsToKeep::add);
        this.taskQueue.add(new BulkDeleteTask(alarmIdsToKeep, this.getCurrentTimeMillis()));
        this.alarmDocumentsById.keySet().removeIf(alarmId -> !alarmIdsToKeep.contains(alarmId));
    }

    public synchronized void postHandleAlarmSnapshot() {
        this.stateTracker.resetStateAndStopTrackingAlarms();
    }

    public synchronized void handleNewOrUpdatedAlarm(OnmsAlarm alarm) {
        LOG.debug("Got new or updated alarm callback for alarm with id: {} and reduction key: {}", (Object)alarm.getId(), (Object)alarm.getReductionKey());
        this.getDocumentIfNeedsIndexing(alarm).ifPresent(a -> {
            this.alarmDocumentsToIndex.add((AlarmDocumentDTO)a);
            if (this.alarmDocumentsToIndex.size() >= this.batchSize) {
                this.flushDocumentsToIndexToTaskQueue();
            }
            this.stateTracker.trackNewOrUpdatedAlarm(alarm.getId().intValue(), alarm.getReductionKey());
        });
    }

    public synchronized void handleDeletedAlarm(int alarmId, String reductionKey) {
        LOG.debug("Got delete callback for alarm with id: {} and reduction key: {}", (Object)alarmId, (Object)reductionKey);
        AlarmDocumentDTO alarmDocument = this.documentFactory.createAlarmDocumentForDelete(alarmId, reductionKey);
        this.alarmDocumentsToIndex.add(alarmDocument);
        if (this.alarmDocumentsToIndex.size() >= this.batchSize) {
            this.flushDocumentsToIndexToTaskQueue();
        }
        this.stateTracker.trackDeletedAlarm(alarmId, reductionKey);
    }

    private synchronized void flushDocumentsToIndexToTaskQueue() {
        if (!this.alarmDocumentsToIndex.isEmpty()) {
            this.taskQueue.add(new IndexAlarmsTask(new ArrayList<AlarmDocumentDTO>(this.alarmDocumentsToIndex)));
            this.alarmDocumentsToIndex.clear();
        }
    }

    private boolean interestingEquals(AlarmDocumentDTO document, OnmsAlarm alarm) {
        Objects.requireNonNull(document);
        Objects.requireNonNull(alarm);
        if (document.getStickyMemo() == null && alarm.getStickyMemo() != null) {
            return false;
        }
        if (document.getJournalMemo() == null && alarm.getReductionKeyMemo() != null) {
            return false;
        }
        return Objects.equals(document.getReductionKey(), alarm.getReductionKey()) && Objects.equals(document.getAckTime(), alarm.getAckTime() == null ? null : Long.valueOf(alarm.getAckTime().getTime())) && Objects.equals(document.getSeverityId(), alarm.getSeverityId()) && Objects.equals(document.getRelatedAlarmIds(), alarm.getRelatedAlarmIds()) && Objects.equals(document.getStickyMemo() == null ? null : document.getStickyMemo().getUpdateTime(), alarm.getStickyMemo() == null ? null : (alarm.getStickyMemo().getUpdated() == null ? null : Long.valueOf(alarm.getStickyMemo().getUpdated().getTime()))) && Objects.equals(document.getJournalMemo() == null ? null : document.getJournalMemo().getUpdateTime(), alarm.getReductionKeyMemo() == null ? null : (alarm.getReductionKeyMemo().getUpdated() == null ? null : Long.valueOf(alarm.getReductionKeyMemo().getUpdated().getTime()))) && Objects.equals(document.getTicketStateId(), alarm.getTTicketState() == null ? null : Integer.valueOf(alarm.getTTicketState().getValue())) && Objects.equals(document.getSituation(), alarm.isSituation());
    }

    @VisibleForTesting
    Optional<AlarmDocumentDTO> getDocumentIfNeedsIndexing(OnmsAlarm alarm) {
        AlarmDocumentDTO existingAlarmDocument = this.alarmDocumentsById.get(alarm.getId());
        boolean needsIndexing = false;
        if (this.indexAllUpdates) {
            needsIndexing = true;
        } else if (existingAlarmDocument == null) {
            needsIndexing = true;
        } else if (this.getCurrentTimeMillis() - existingAlarmDocument.getUpdateTime() >= this.alarmReindexDurationMs) {
            needsIndexing = true;
        } else if (!this.interestingEquals(existingAlarmDocument, alarm)) {
            needsIndexing = true;
        }
        if (needsIndexing) {
            AlarmDocumentDTO doc;
            try {
                doc = this.documentMapper.apply(alarm);
            }
            catch (Exception e) {
                LOG.warn("Mapping alarm to DTO failed. Document will not be indexed.", (Throwable)e);
                return Optional.empty();
            }
            this.alarmDocumentsById.put(alarm.getId(), doc);
            return Optional.of(doc);
        }
        return Optional.empty();
    }

    private long getCurrentTimeMillis() {
        if (this.usePseudoClock) {
            return PseudoClock.getInstance().getTime();
        }
        return System.currentTimeMillis();
    }

    public void setBulkRetryCount(int bulkRetryCount) {
        this.bulkRetryCount = bulkRetryCount;
    }

    public void setBatchSize(int batchSize) {
        this.batchSize = batchSize;
    }

    public void setAlarmReindexDurationMs(long alarmReindexDurationMs) {
        this.alarmReindexDurationMs = alarmReindexDurationMs;
    }

    public void setLookbackPeriodMs(long lookbackPeriodMs) {
        this.lookbackPeriodMs = lookbackPeriodMs;
    }

    public void setUsePseudoClock(boolean usePseudoClock) {
        this.usePseudoClock = usePseudoClock;
    }

    public void setIndexAllUpdates(boolean indexAllUpdates) {
        this.indexAllUpdates = indexAllUpdates;
    }
}

