/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.dao.hibernate;

import static org.opennms.core.utils.InetAddressUtils.str;

import java.net.InetAddress;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Timer;
import java.util.TimerTask;
import java.util.TreeSet;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.opennms.core.criteria.CriteriaBuilder;
import org.opennms.core.utils.LocationUtils;
import org.opennms.netmgt.dao.api.AbstractInterfaceToNodeCache;
import org.opennms.netmgt.dao.api.InterfaceToNodeCache;
import org.opennms.netmgt.dao.api.IpInterfaceDao;
import org.opennms.netmgt.dao.api.NodeDao;
import org.opennms.netmgt.model.OnmsIpInterface;
import org.opennms.netmgt.model.OnmsNode;
import org.opennms.netmgt.model.PrimaryType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionOperations;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import com.google.common.collect.ComparisonChain;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimaps;
import com.google.common.collect.SortedSetMultimap;
import com.google.common.util.concurrent.ThreadFactoryBuilder;

/**
 * This class represents a singular instance that is used to map IP
 * addresses to known nodes.
 *
 * @author Seth
 * @author <a href="mailto:joed@opennms.org">Johan Edstrom</a>
 * @author <a href="mailto:weave@oculan.com">Brian Weaver </a>
 * @author <a href="mailto:tarus@opennms.org">Tarus Balog </a>
 * @author <a href="http://www.opennms.org/">OpenNMS </a>
 */
public class InterfaceToNodeCacheDaoImpl extends AbstractInterfaceToNodeCache implements InterfaceToNodeCache {
    private static final Logger LOG = LoggerFactory.getLogger(InterfaceToNodeCacheDaoImpl.class);

    private final ThreadFactory threadFactory = new ThreadFactoryBuilder()
            .setNameFormat("sync-interface-to-node-cache")
            .build();
    private final ExecutorService executorService = Executors.newSingleThreadExecutor(threadFactory);
    private final CountDownLatch initialNodeSyncDone = new CountDownLatch(1);

    private static class Key {
        private final String location;
        private final InetAddress ipAddress;

        public Key(String location, InetAddress ipAddress) {
            // Use the default location when location is null
            this.location = LocationUtils.getEffectiveLocationName(location);
            this.ipAddress = Objects.requireNonNull(ipAddress);
        }

        public InetAddress getIpAddress() {
            return ipAddress;
        }

        public String getLocation() {
            return location;
        }

        @Override
        public boolean equals(final Object obj) {
            if (obj == null) {
                return false;
            }
            if (obj == this) {
                return true;
            }
            if (obj.getClass() != getClass()) {
                return false;
            }
            final Key that = (Key) obj;
            return Objects.equals(this.ipAddress, that.ipAddress)
                    && Objects.equals(this.location, that.location);
        }

        @Override
        public int hashCode() {
            return Objects.hash(this.ipAddress, this.location);
        }

        @Override
        public String toString() {
            return String.format("Key[location='%s', ipAddress='%s']", this.location, this.ipAddress);
        }
    }

    private static class Value implements Comparable<Value> {
        private final int nodeId;
        private final int interfaceId;
        private final PrimaryType type;


        private Value(final int nodeId,
                      final int interfaceId,
                      final PrimaryType type) {
            this.nodeId = nodeId;
            this.interfaceId = interfaceId;
            this.type = type;
        }

        public int getNodeId() {
            return this.nodeId;
        }

        public int getInterfaceId() {
            return this.interfaceId;
        }

        public PrimaryType getType() {
            return this.type;
        }

        @Override
        public boolean equals(final Object obj) {
            if (obj == null) {
                return false;
            }
            if (obj == this) {
                return true;
            }
            if (obj.getClass() != getClass()) {
                return false;
            }
            final Value that = (Value) obj;
            return Objects.equals(this.nodeId, that.nodeId)
                    && Objects.equals(this.interfaceId, that.interfaceId)
                    && Objects.equals(this.type, that.type);
        }

        @Override
        public int hashCode() {
            return Objects.hash(this.nodeId, this.type.getCharCode());
        }

        @Override
        public String toString() {
            return String.format("Value[nodeId='%s', interfaceId='%s', type='%s']", this.nodeId, this.interfaceId, this.type);
        }

        @Override
        public int compareTo(final Value that) {
            return ComparisonChain.start()
                    .compare(this.type, that.type)
                    .compare(this.nodeId, that.nodeId)
                    .compare(this.interfaceId, that.interfaceId)
                    .result();
        }
    }

    @Autowired
    private NodeDao m_nodeDao;

    @Autowired
    private IpInterfaceDao m_ipInterfaceDao;

    @Autowired
    private TransactionOperations transactionOperations;

    private final ReadWriteLock m_lock = new ReentrantReadWriteLock();
    private SortedSetMultimap<Key, Value> m_managedAddresses = Multimaps.newSortedSetMultimap(Maps.newHashMap(), TreeSet::new);

    private final Timer refreshTimer = new Timer(getClass().getSimpleName());

    // in ms
    private final long refreshRate;

    public InterfaceToNodeCacheDaoImpl() {
        this(-1); // By default refreshing the cache is disabled
    }

    public InterfaceToNodeCacheDaoImpl(long refreshRate) {
        this.refreshRate = refreshRate;
    }

    @PostConstruct
    public void init() {
        // sync datasource asynchronously in order to not block bean initialization.
        syncDataSourceAsynchronously().whenComplete((result, ex) -> {
            initialNodeSyncDone.countDown();
            if (refreshRate > 0) {
                refreshTimer.schedule(new TimerTask() {
                    @Override
                    public void run() {
                        try {
                            dataSourceSync();
                        } catch (Exception ex) {
                            LOG.error("An error occurred while synchronizing the datasource: {}", ex.getMessage(), ex);
                        }
                    }
                }, refreshRate, refreshRate);
            }
        });
    }

    @PreDestroy
    public void destroy() {
        initialNodeSyncDone.countDown();
        executorService.shutdownNow();
    }

    public NodeDao getNodeDao() {
        return m_nodeDao;
    }

    public void setNodeDao(NodeDao nodeDao) {
        m_nodeDao = nodeDao;
    }

    public IpInterfaceDao getIpInterfaceDao() {
        return m_ipInterfaceDao;
    }

    public void setIpInterfaceDao(IpInterfaceDao ipInterfaceDao) {
        m_ipInterfaceDao = ipInterfaceDao;
    }

    /**
     * Clears and synchronizes the internal known IP address cache with the
     * current information contained in the database. To synchronize the cache
     * the method opens a new connection to the database, loads the address,
     * and then closes it's connection.
     *
     * @throws java.sql.SQLException Thrown if the connection cannot be created or a database
     *                               error occurs.
     */
    @Override
    @Transactional
    public void dataSourceSync() {
        // Determine if spring already created a transaction or if we have to manually do it
        // (depending on where the call to dataSourceSync comes from)
        if (TransactionSynchronizationManager.isActualTransactionActive()) {
            dataSourceSyncWithinTransaction();
        } else {
            transactionOperations.execute((status) -> {
                dataSourceSyncWithinTransaction();
                return null;
            });
        }
    }

    private CompletableFuture<Void> syncDataSourceAsynchronously() {
        return CompletableFuture.runAsync(this::dataSourceSync, executorService);
    }

    private void dataSourceSyncWithinTransaction() {
        /*
         * Make a new list with which we'll replace the existing one, that way
         * if something goes wrong with the DB we won't lose whatever was already
         * in there
         */
        final SortedSetMultimap<Key, Value> newAlreadyDiscovered = Multimaps.newSortedSetMultimap(Maps.newHashMap(), TreeSet::new);

        // Fetch all non-deleted nodes
        final CriteriaBuilder builder = new CriteriaBuilder(OnmsNode.class);
        builder.ne("type", String.valueOf(OnmsNode.NodeType.DELETED.value()));

        for (OnmsNode node : m_nodeDao.findMatching(builder.toCriteria())) {
            for (final OnmsIpInterface iface : node.getIpInterfaces()) {
                // Skip deleted interfaces
                // TODO: Refactor the 'D' value with an enumeration
                if ("D".equals(iface.getIsManaged())) {
                    continue;
                }
                LOG.debug("Adding entry: {}:{} -> {}", node.getLocation().getLocationName(), iface.getIpAddress(), node.getId());
                newAlreadyDiscovered.put(new Key(node.getLocation().getLocationName(), iface.getIpAddress()), new Value(node.getId(), iface.getId(), iface.getIsSnmpPrimary()));
            }
        }

        try {
            m_lock.writeLock().lock();
            m_managedAddresses = newAlreadyDiscovered;
        } finally {
            m_lock.writeLock().unlock();
        }

        LOG.info("dataSourceSync: initialized list of managed IP addresses with {} members", m_managedAddresses.size());
    }

    @Override
    public Optional<Entry> getFirst(String location, InetAddress ipAddr) {
        if (ipAddr == null) {
            return Optional.empty();
        }
        waitForInitialNodeSync();
        m_lock.readLock().lock();
        try {
            var values = m_managedAddresses.get(new Key(location, ipAddr));
            return values.isEmpty() ? Optional.empty() : Optional.of(new Entry(values.first().nodeId, values.first().interfaceId));
        } finally {
            m_lock.readLock().unlock();
        }
    }

    private void waitForInitialNodeSync() {
        try {
            initialNodeSyncDone.await();
        } catch (InterruptedException e) {
            LOG.warn("Wait for node cache sync interrupted", e);
        }
    }

    /**
     * Sets the IP Address and Node ID in the Map.
     *
     * @param addr   The IP Address to add.
     * @param nodeid The Node ID to add.
     * @return The nodeid if it existed in the map.
     */
    @Override
    @Transactional
    public boolean setNodeId(final String location, final InetAddress addr, final int nodeid) {
        if (addr == null || nodeid == -1) {
            return false;
        }

        final OnmsIpInterface iface = m_ipInterfaceDao.findByNodeIdAndIpAddress(nodeid, str(addr));
        if (iface == null) {
            return false;
        }

        LOG.debug("setNodeId: adding IP address to cache: {}:{} -> {}", location, str(addr), nodeid);

        m_lock.writeLock().lock();
        try {
            return m_managedAddresses.put(new Key(location, addr), new Value(nodeid, iface.getId(), iface.getIsSnmpPrimary()));
        } finally {
            m_lock.writeLock().unlock();
        }
    }

    /**
     * Removes an address from the node ID map.
     *
     * @param address The address to remove from the node ID map.
     * @return The nodeid that was in the map.
     */
    @Override
    public boolean removeNodeId(final String location, final InetAddress address, final int nodeId) {
        if (address == null) {
            LOG.warn("removeNodeId: null IP address");
            return false;
        }

        LOG.debug("removeNodeId: removing IP address from cache: {}:{}", location, str(address));

        m_lock.writeLock().lock();
        try {
            final Key key = new Key(location, address);
            return m_managedAddresses.get(key).removeIf(e -> e.nodeId == nodeId);
        } finally {
            m_lock.writeLock().unlock();
        }
    }

    @Override
    public int size() {
        waitForInitialNodeSync();
        m_lock.readLock().lock();
        try {
            return m_managedAddresses.size();
        } finally {
            m_lock.readLock().unlock();
        }
    }

    @Override
    public void clear() {
        m_lock.writeLock().lock();
        try {
            m_managedAddresses.clear();
        } finally {
            m_lock.writeLock().unlock();
        }
    }

    @Override
    public void removeInterfacesForNode(int nodeId) {
        m_lock.writeLock().lock();
        try {
            List<Map.Entry<Key, Value>> keyValues = m_managedAddresses.entries().stream()
                    .filter(keyValueEntry -> keyValueEntry.getValue().getNodeId() == nodeId)
                    .collect(Collectors.toList());
            keyValues.forEach(keyValue -> {
                boolean succeeded = m_managedAddresses.remove(keyValue.getKey(), keyValue.getValue());
                if (succeeded) {
                    LOG.debug("removeInterfacesForNode: removed IP address from cache: {}", str(keyValue.getKey().getIpAddress()));
                }
            });
        } finally {
            m_lock.writeLock().unlock();
        }
    }
}
