/*
 * Decompiled with CFR 0.152.
 */
package org.jrobin.core;

import java.io.IOException;
import org.jrobin.core.Archive;
import org.jrobin.core.DataImporter;
import org.jrobin.core.DsDef;
import org.jrobin.core.DsTypes;
import org.jrobin.core.Header;
import org.jrobin.core.RrdAllocator;
import org.jrobin.core.RrdBackend;
import org.jrobin.core.RrdDb;
import org.jrobin.core.RrdDouble;
import org.jrobin.core.RrdException;
import org.jrobin.core.RrdLong;
import org.jrobin.core.RrdString;
import org.jrobin.core.RrdUpdater;
import org.jrobin.core.Util;
import org.jrobin.core.XmlWriter;

public class Datasource
implements RrdUpdater,
DsTypes {
    private static final double MAX_32_BIT = Math.pow(2.0, 32.0);
    private static final double MAX_64_BIT = Math.pow(2.0, 64.0);
    private RrdDb parentDb;
    private RrdString dsName;
    private RrdString dsType;
    private RrdLong heartbeat;
    private RrdDouble minValue;
    private RrdDouble maxValue;
    private String m_primitiveDsName = null;
    private String m_primitiveDsType = null;
    private RrdDouble lastValue;
    private RrdLong nanSeconds;
    private RrdDouble accumValue;

    Datasource(RrdDb parentDb, DsDef dsDef) throws IOException {
        boolean shouldInitialize = dsDef != null;
        this.parentDb = parentDb;
        this.dsName = new RrdString(this);
        this.dsType = new RrdString(this);
        this.heartbeat = new RrdLong(this);
        this.minValue = new RrdDouble(this);
        this.maxValue = new RrdDouble(this);
        this.lastValue = new RrdDouble(this);
        this.accumValue = new RrdDouble(this);
        this.nanSeconds = new RrdLong(this);
        if (shouldInitialize) {
            this.dsName.set(dsDef.getDsName());
            this.m_primitiveDsName = null;
            this.dsType.set(dsDef.getDsType());
            this.m_primitiveDsType = null;
            this.heartbeat.set(dsDef.getHeartbeat());
            this.minValue.set(dsDef.getMinValue());
            this.maxValue.set(dsDef.getMaxValue());
            this.lastValue.set(Double.NaN);
            this.accumValue.set(0.0);
            Header header = parentDb.getHeader();
            this.nanSeconds.set(header.getLastUpdateTime() % header.getStep());
        }
    }

    Datasource(RrdDb parentDb, DataImporter reader, int dsIndex) throws IOException, RrdException {
        this(parentDb, null);
        this.dsName.set(reader.getDsName(dsIndex));
        this.m_primitiveDsName = null;
        this.dsType.set(reader.getDsType(dsIndex));
        this.m_primitiveDsType = null;
        this.heartbeat.set(reader.getHeartbeat(dsIndex));
        this.minValue.set(reader.getMinValue(dsIndex));
        this.maxValue.set(reader.getMaxValue(dsIndex));
        this.lastValue.set(reader.getLastValue(dsIndex));
        this.accumValue.set(reader.getAccumValue(dsIndex));
        this.nanSeconds.set(reader.getNanSeconds(dsIndex));
    }

    String dump() throws IOException {
        return "== DATASOURCE ==\nDS:" + this.dsName.get() + ":" + this.dsType.get() + ":" + this.heartbeat.get() + ":" + this.minValue.get() + ":" + this.maxValue.get() + "\nlastValue:" + this.lastValue.get() + " nanSeconds:" + this.nanSeconds.get() + " accumValue:" + this.accumValue.get() + "\n";
    }

    public String getDsName() throws IOException {
        if (this.m_primitiveDsName == null) {
            this.m_primitiveDsName = this.dsName.get();
        }
        return this.m_primitiveDsName;
    }

    public String getDsType() throws IOException {
        if (this.m_primitiveDsType == null) {
            this.m_primitiveDsType = this.dsType.get();
        }
        return this.m_primitiveDsType;
    }

    public long getHeartbeat() throws IOException {
        return this.heartbeat.get();
    }

    public double getMinValue() throws IOException {
        return this.minValue.get();
    }

    public double getMaxValue() throws IOException {
        return this.maxValue.get();
    }

    public double getLastValue() throws IOException {
        return this.lastValue.get();
    }

    public double getAccumValue() throws IOException {
        return this.accumValue.get();
    }

    public long getNanSeconds() throws IOException {
        return this.nanSeconds.get();
    }

    void process(long newTime, double newValue) throws IOException, RrdException {
        Header header = this.parentDb.getHeader();
        long step = header.getStep();
        long oldTime = header.getLastUpdateTime();
        long startTime = Util.normalize(oldTime, step);
        long endTime = startTime + step;
        double oldValue = this.lastValue.get();
        double updateValue = this.calculateUpdateValue(oldTime, oldValue, newTime, newValue);
        if (newTime < endTime) {
            this.accumulate(oldTime, newTime, updateValue);
        } else {
            long boundaryTime = Util.normalize(newTime, step);
            this.accumulate(oldTime, boundaryTime, updateValue);
            double value = this.calculateTotal(startTime, boundaryTime);
            long numSteps = (boundaryTime - endTime) / step + 1L;
            this.parentDb.archive(this, value, numSteps);
            this.nanSeconds.set(0L);
            this.accumValue.set(0.0);
            this.accumulate(boundaryTime, newTime, updateValue);
        }
    }

    private double calculateUpdateValue(long oldTime, double oldValue, long newTime, double newValue) throws IOException {
        double updateValue = Double.NaN;
        if (newTime - oldTime <= this.heartbeat.get()) {
            String type = this.dsType.get();
            if (type.equals("GAUGE")) {
                updateValue = newValue;
            } else if (type.equals("ABSOLUTE")) {
                if (!Double.isNaN(newValue)) {
                    updateValue = newValue / (double)(newTime - oldTime);
                }
            } else if (type.equals("DERIVE")) {
                if (!Double.isNaN(newValue) && !Double.isNaN(oldValue)) {
                    updateValue = (newValue - oldValue) / (double)(newTime - oldTime);
                }
            } else if (type.equals("COUNTER") && !Double.isNaN(newValue) && !Double.isNaN(oldValue)) {
                double diff = newValue - oldValue;
                if (diff < 0.0) {
                    diff += MAX_32_BIT;
                }
                if (diff < 0.0) {
                    diff += MAX_64_BIT - MAX_32_BIT;
                }
                if (diff >= 0.0) {
                    updateValue = diff / (double)(newTime - oldTime);
                }
            }
            if (!Double.isNaN(updateValue)) {
                double minVal = this.minValue.get();
                double maxVal = this.maxValue.get();
                if (!Double.isNaN(minVal) && updateValue < minVal) {
                    updateValue = Double.NaN;
                }
                if (!Double.isNaN(maxVal) && updateValue > maxVal) {
                    updateValue = Double.NaN;
                }
            }
        }
        this.lastValue.set(newValue);
        return updateValue;
    }

    private void accumulate(long oldTime, long newTime, double updateValue) throws IOException {
        if (Double.isNaN(updateValue)) {
            this.nanSeconds.set(this.nanSeconds.get() + (newTime - oldTime));
        } else {
            this.accumValue.set(this.accumValue.get() + updateValue * (double)(newTime - oldTime));
        }
    }

    private double calculateTotal(long startTime, long boundaryTime) throws IOException {
        double totalValue = Double.NaN;
        long validSeconds = boundaryTime - startTime - this.nanSeconds.get();
        if (this.nanSeconds.get() <= this.heartbeat.get() && validSeconds > 0L) {
            totalValue = this.accumValue.get() / (double)validSeconds;
        }
        if (Double.isNaN(totalValue) && this.dsName.get().endsWith("!")) {
            totalValue = 0.0;
        }
        return totalValue;
    }

    void appendXml(XmlWriter writer) throws IOException {
        writer.startTag("ds");
        writer.writeTag("name", this.dsName.get());
        writer.writeTag("type", this.dsType.get());
        writer.writeTag("minimal_heartbeat", this.heartbeat.get());
        writer.writeTag("min", this.minValue.get());
        writer.writeTag("max", this.maxValue.get());
        writer.writeComment("PDP Status");
        writer.writeTag("last_ds", this.lastValue.get(), "UNKN");
        writer.writeTag("value", this.accumValue.get());
        writer.writeTag("unknown_sec", this.nanSeconds.get());
        writer.closeTag();
    }

    @Override
    public void copyStateTo(RrdUpdater other) throws IOException, RrdException {
        if (!(other instanceof Datasource)) {
            throw new RrdException("Cannot copy Datasource object to " + other.getClass().getName());
        }
        Datasource datasource = (Datasource)other;
        if (!datasource.dsName.get().equals(this.dsName.get())) {
            throw new RrdException("Incomaptible datasource names");
        }
        if (!datasource.dsType.get().equals(this.dsType.get())) {
            throw new RrdException("Incomaptible datasource types");
        }
        datasource.lastValue.set(this.lastValue.get());
        datasource.nanSeconds.set(this.nanSeconds.get());
        datasource.accumValue.set(this.accumValue.get());
    }

    public int getDsIndex() throws IOException {
        try {
            return this.parentDb.getDsIndex(this.dsName.get());
        }
        catch (RrdException e) {
            return -1;
        }
    }

    public void setHeartbeat(long heartbeat) throws RrdException, IOException {
        if (heartbeat < 1L) {
            throw new RrdException("Invalid heartbeat specified: " + heartbeat);
        }
        this.heartbeat.set(heartbeat);
    }

    public void setDsName(String newDsName) throws RrdException, IOException {
        if (newDsName.length() > 20) {
            throw new RrdException("Invalid datasource name specified: " + newDsName);
        }
        if (this.parentDb.containsDs(newDsName)) {
            throw new RrdException("Datasource already defined in this RRD: " + newDsName);
        }
        this.dsName.set(newDsName);
        this.m_primitiveDsName = null;
    }

    public void setDsType(String newDsType) throws RrdException, IOException {
        Archive[] archives;
        if (!DsDef.isValidDsType(newDsType)) {
            throw new RrdException("Invalid datasource type: " + newDsType);
        }
        this.dsType.set(newDsType);
        this.m_primitiveDsType = null;
        this.lastValue.set(Double.NaN);
        this.accumValue.set(0.0);
        int dsIndex = this.parentDb.getDsIndex(this.dsName.get());
        for (Archive archive : archives = this.parentDb.getArchives()) {
            archive.getArcState(dsIndex).setAccumValue(Double.NaN);
        }
    }

    public void setMinValue(double minValue, boolean filterArchivedValues) throws IOException, RrdException {
        double maxValue = this.maxValue.get();
        if (!Double.isNaN(minValue) && !Double.isNaN(maxValue) && minValue >= maxValue) {
            throw new RrdException("Invalid min/max values: " + minValue + "/" + maxValue);
        }
        this.minValue.set(minValue);
        if (!Double.isNaN(minValue) && filterArchivedValues) {
            Archive[] archives;
            int dsIndex = this.getDsIndex();
            for (Archive archive : archives = this.parentDb.getArchives()) {
                archive.getRobin(dsIndex).filterValues(minValue, Double.NaN);
            }
        }
    }

    public void setMaxValue(double maxValue, boolean filterArchivedValues) throws IOException, RrdException {
        double minValue = this.minValue.get();
        if (!Double.isNaN(minValue) && !Double.isNaN(maxValue) && minValue >= maxValue) {
            throw new RrdException("Invalid min/max values: " + minValue + "/" + maxValue);
        }
        this.maxValue.set(maxValue);
        if (!Double.isNaN(maxValue) && filterArchivedValues) {
            Archive[] archives;
            int dsIndex = this.getDsIndex();
            for (Archive archive : archives = this.parentDb.getArchives()) {
                archive.getRobin(dsIndex).filterValues(Double.NaN, maxValue);
            }
        }
    }

    public void setMinMaxValue(double minValue, double maxValue, boolean filterArchivedValues) throws IOException, RrdException {
        if (!Double.isNaN(minValue) && !Double.isNaN(maxValue) && minValue >= maxValue) {
            throw new RrdException("Invalid min/max values: " + minValue + "/" + maxValue);
        }
        this.minValue.set(minValue);
        this.maxValue.set(maxValue);
        if (!(Double.isNaN(minValue) && Double.isNaN(maxValue) || !filterArchivedValues)) {
            Archive[] archives;
            int dsIndex = this.getDsIndex();
            for (Archive archive : archives = this.parentDb.getArchives()) {
                archive.getRobin(dsIndex).filterValues(minValue, maxValue);
            }
        }
    }

    @Override
    public RrdBackend getRrdBackend() {
        return this.parentDb.getRrdBackend();
    }

    @Override
    public RrdAllocator getRrdAllocator() {
        return this.parentDb.getRrdAllocator();
    }

    public String toString() {
        return this.getClass().getName() + "@" + Integer.toHexString(this.hashCode()) + "[parentDb=" + this.parentDb + ",dsName=" + this.dsName + ",dsType=" + this.dsType + ",heartbeat=" + this.heartbeat + ",minValue=" + this.minValue + ",maxValue=" + this.maxValue + "]";
    }
}

