/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.core.ipc.twin.common;

import com.codahale.metrics.Counter;
import com.codahale.metrics.MetricRegistry;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.fge.jsonpatch.diff.JsonDiff;
import com.google.common.base.Strings;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import io.opentracing.Span;
import io.opentracing.Tracer;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiConsumer;
import org.opennms.core.ipc.twin.api.TwinPublisher;
import org.opennms.core.ipc.twin.common.LocalTwinSubscriber;
import org.opennms.core.ipc.twin.common.TwinRequest;
import org.opennms.core.ipc.twin.common.TwinTracker;
import org.opennms.core.ipc.twin.common.TwinUpdate;
import org.opennms.core.ipc.twin.model.TwinRequestProto;
import org.opennms.core.ipc.twin.model.TwinResponseProto;
import org.opennms.core.logging.Logging;
import org.opennms.core.tracing.api.TracerRegistry;
import org.opennms.core.tracing.util.TracingInfoCarrier;
import org.opennms.core.utils.SystemInfoUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractTwinPublisher
implements TwinPublisher {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractTwinPublisher.class);
    protected static final String TAG_TWIN_SINK = "TwinSink";
    protected static final String TAG_TWIN_RPC = "TwinRPC";
    protected static final String TAG_VERSION = "version";
    protected static final String TAG_SESSION_ID = "sessionId";
    protected static final String TAG_PATCH = "isPatch";
    private static final String SINK_UPDATE_SENT = "sinkUpdateSent";
    private static final String TWIN_RESPONSE_SENT = "twinResponseSent";
    private static final String TWIN_EMPTY_RESPONSE_SENT = "twinEmptyResponseSent";
    private final Map<SessionKey, TwinTracker> twinTrackerMap = new HashMap<SessionKey, TwinTracker>();
    protected final ObjectMapper objectMapper = new ObjectMapper();
    private final LocalTwinSubscriber localTwinSubscriber;
    private final Tracer tracer;
    private final MetricRegistry metrics;

    public AbstractTwinPublisher(LocalTwinSubscriber localTwinSubscriber, TracerRegistry tracerRegistry, MetricRegistry metricRegistry) {
        this.localTwinSubscriber = Objects.requireNonNull(localTwinSubscriber);
        Objects.requireNonNull(tracerRegistry);
        tracerRegistry.init(SystemInfoUtils.getInstanceId());
        this.tracer = tracerRegistry.getTracer();
        this.metrics = metricRegistry;
    }

    public AbstractTwinPublisher(LocalTwinSubscriber localTwinSubscriber) {
        this(localTwinSubscriber, localTwinSubscriber.getTracerRegistry(), localTwinSubscriber.getMetricRegistry());
    }

    protected abstract void handleSinkUpdate(TwinUpdate var1);

    public <T> TwinPublisher.Session<T> register(String key, Class<T> clazz, String location) throws IOException {
        try (Logging.MDCCloseable mdc = Logging.withPrefixCloseable((String)"ipc");){
            SessionKey sessionKey = new SessionKey(key, location);
            LOG.info("Registered a session with key {}", (Object)sessionKey);
            SessionImpl sessionImpl = new SessionImpl(sessionKey);
            return sessionImpl;
        }
    }

    protected TwinUpdate getTwin(TwinRequest twinRequest) {
        TwinUpdate twinUpdate;
        TwinTracker twinTracker = this.getTwinTracker(twinRequest.getKey(), twinRequest.getLocation());
        if (twinTracker == null) {
            twinUpdate = new TwinUpdate(twinRequest.getKey(), twinRequest.getLocation(), null);
            this.updateCounter(MetricRegistry.name((String)twinRequest.location, (String[])new String[]{twinRequest.getKey(), TWIN_EMPTY_RESPONSE_SENT}));
        } else {
            twinUpdate = new TwinUpdate(twinRequest.getKey(), twinRequest.getLocation(), twinTracker.getObj());
            twinUpdate.setPatch(false);
            twinUpdate.setVersion(twinTracker.getVersion());
            twinUpdate.setSessionId(twinTracker.getSessionId());
            this.updateCounter(MetricRegistry.name((String)twinRequest.location, (String[])new String[]{twinRequest.getKey(), TWIN_RESPONSE_SENT}));
        }
        return twinUpdate;
    }

    private synchronized TwinTracker getTwinTracker(String key, String location) {
        TwinTracker twinTracker = this.twinTrackerMap.get(new SessionKey(key, location));
        if (twinTracker == null) {
            twinTracker = this.twinTrackerMap.get(new SessionKey(key, null));
        }
        return twinTracker;
    }

    protected TwinResponseProto mapTwinResponse(TwinUpdate twinUpdate) {
        TwinResponseProto.Builder builder = TwinResponseProto.newBuilder();
        if (!Strings.isNullOrEmpty((String)twinUpdate.getLocation())) {
            builder.setLocation(twinUpdate.getLocation());
        }
        if (!Strings.isNullOrEmpty((String)twinUpdate.getSessionId())) {
            builder.setSessionId(twinUpdate.getSessionId());
        }
        builder.setConsumerKey(twinUpdate.getKey());
        if (twinUpdate.getObject() != null) {
            builder.setTwinObject(ByteString.copyFrom((byte[])twinUpdate.getObject()));
        }
        builder.setIsPatchObject(twinUpdate.isPatch());
        builder.setVersion(twinUpdate.getVersion());
        twinUpdate.getTracingInfo().forEach(builder::putTracingInfo);
        return builder.build();
    }

    protected TwinRequest mapTwinRequestProto(byte[] twinRequestBytes) {
        TwinRequest twinRequest = new TwinRequest();
        try {
            TwinRequestProto twinRequestProto = TwinRequestProto.parseFrom(twinRequestBytes);
            twinRequest.setKey(twinRequestProto.getConsumerKey());
            if (!Strings.isNullOrEmpty((String)twinRequestProto.getLocation())) {
                twinRequest.setLocation(twinRequestProto.getLocation());
            }
            twinRequestProto.getTracingInfoMap().forEach(twinRequest::addTracingInfo);
        }
        catch (InvalidProtocolBufferException e) {
            LOG.warn("Failed to parse protobuf for the request", (Throwable)e);
            throw new RuntimeException(e);
        }
        return twinRequest;
    }

    protected void addTracingInfo(Span span, TwinUpdate twinUpdate) {
        TracingInfoCarrier.updateTracingMetadata((Tracer)this.getTracer(), (Span)span, twinUpdate::addTracingInfo);
        span.setTag(TAG_TWIN_RPC, true);
        span.setTag(TAG_VERSION, (Number)twinUpdate.getVersion());
        span.setTag(TAG_SESSION_ID, twinUpdate.getSessionId());
        span.setTag(TAG_PATCH, twinUpdate.isPatch());
    }

    private void updateCounter(String counterName) {
        Counter counter = this.metrics.counter(counterName);
        counter.inc();
    }

    public static String generateTracingOperationKey(String location, String key) {
        return location != null ? key + "@" + location : key;
    }

    private synchronized TwinUpdate getTwinUpdateFromUpdatedObj(byte[] updatedObj, SessionKey sessionKey) {
        TwinTracker twinTracker = this.getTwinTracker(sessionKey.key, sessionKey.location);
        if (twinTracker == null || !Arrays.equals(twinTracker.getObj(), updatedObj)) {
            TwinUpdate twinUpdate = new TwinUpdate(sessionKey.key, sessionKey.location, updatedObj);
            if (twinTracker == null) {
                twinTracker = new TwinTracker(updatedObj);
            } else {
                byte[] patchValue = this.getPatchValue(twinTracker.getObj(), updatedObj, sessionKey);
                if (patchValue != null) {
                    twinUpdate.setObject(patchValue);
                    twinUpdate.setPatch(true);
                }
                twinTracker.update(updatedObj);
            }
            this.twinTrackerMap.put(sessionKey, twinTracker);
            twinUpdate.setVersion(twinTracker.getVersion());
            twinUpdate.setSessionId(twinTracker.getSessionId());
            return twinUpdate;
        }
        return null;
    }

    private byte[] getPatchValue(byte[] originalObj, byte[] updatedObj, SessionKey sessionKey) {
        try {
            JsonNode sourceNode = this.objectMapper.readTree(originalObj);
            JsonNode targetNode = this.objectMapper.readTree(updatedObj);
            JsonNode diffNode = JsonDiff.asJson((JsonNode)sourceNode, (JsonNode)targetNode);
            return diffNode.toString().getBytes(StandardCharsets.UTF_8);
        }
        catch (Exception e) {
            LOG.error("Unable to generate patch for SessionKey {}", (Object)sessionKey, (Object)e);
            return null;
        }
    }

    private synchronized void removeSessionKey(SessionKey sessionKey) {
        this.twinTrackerMap.remove(sessionKey);
    }

    public synchronized void forEachSession(BiConsumer<SessionKey, TwinTracker> consumer) {
        this.twinTrackerMap.forEach(consumer);
    }

    public Tracer getTracer() {
        return this.tracer;
    }

    public static class SessionKey {
        public final String key;
        public final String location;

        private SessionKey(String key, String location) {
            this.key = key;
            this.location = location;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            SessionKey that = (SessionKey)o;
            return Objects.equals(this.key, that.key) && Objects.equals(this.location, that.location);
        }

        public int hashCode() {
            return Objects.hash(this.key, this.location);
        }

        public String toString() {
            return "SessionKey{key='" + this.key + "', location='" + this.location + "'}";
        }
    }

    private class SessionImpl<T>
    implements TwinPublisher.Session<T> {
        private final SessionKey sessionKey;

        public SessionImpl(SessionKey sessionKey) {
            this.sessionKey = sessionKey;
        }

        public void publish(T obj) throws IOException {
            try (Logging.MDCCloseable mdc = Logging.withPrefixCloseable((String)"ipc");){
                LOG.info("Published an object update for the session with key {}", (Object)this.sessionKey.toString());
                String tracingOperationKey = AbstractTwinPublisher.generateTracingOperationKey(this.sessionKey.location, this.sessionKey.key);
                Span span = AbstractTwinPublisher.this.tracer.buildSpan(tracingOperationKey).start();
                byte[] objInBytes = AbstractTwinPublisher.this.objectMapper.writeValueAsBytes(obj);
                TwinUpdate twinUpdate = AbstractTwinPublisher.this.getTwinUpdateFromUpdatedObj(objInBytes, this.sessionKey);
                if (twinUpdate != null) {
                    TracingInfoCarrier.updateTracingMetadata((Tracer)AbstractTwinPublisher.this.tracer, (Span)span, twinUpdate::addTracingInfo);
                    span.setTag(AbstractTwinPublisher.TAG_TWIN_SINK, true);
                    if (this.sessionKey.location != null) {
                        span.setTag("location", this.sessionKey.location);
                    }
                    span.setTag(AbstractTwinPublisher.TAG_VERSION, (Number)twinUpdate.getVersion());
                    span.setTag(AbstractTwinPublisher.TAG_SESSION_ID, twinUpdate.getSessionId());
                    AbstractTwinPublisher.this.handleSinkUpdate(twinUpdate);
                    String sinkUpdateMetricName = this.sessionKey.location != null ? MetricRegistry.name((String)this.sessionKey.location, (String[])new String[]{this.sessionKey.key, AbstractTwinPublisher.SINK_UPDATE_SENT}) : MetricRegistry.name((String)this.sessionKey.key, (String[])new String[]{AbstractTwinPublisher.SINK_UPDATE_SENT});
                    AbstractTwinPublisher.this.localTwinSubscriber.accept(twinUpdate);
                    AbstractTwinPublisher.this.updateCounter(sinkUpdateMetricName);
                }
                span.finish();
            }
        }

        public void close() throws IOException {
            AbstractTwinPublisher.this.removeSessionKey(this.sessionKey);
            LOG.info("Closed session with key {} ", (Object)this.sessionKey);
        }
    }
}

