/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.netmgt.flows.processing.impl;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import com.google.common.base.Strings;
import com.google.common.cache.CacheLoader;
import java.io.Serializable;
import java.net.InetAddress;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.opennms.core.cache.Cache;
import org.opennms.core.cache.CacheBuilder;
import org.opennms.core.cache.CacheConfig;
import org.opennms.core.cache.CacheConfigBuilder;
import org.opennms.core.mate.api.ContextKey;
import org.opennms.core.utils.InetAddressUtils;
import org.opennms.integration.api.v1.flows.Flow;
import org.opennms.netmgt.dao.api.InterfaceToNodeCache;
import org.opennms.netmgt.dao.api.IpInterfaceDao;
import org.opennms.netmgt.dao.api.NodeDao;
import org.opennms.netmgt.dao.api.SessionUtils;
import org.opennms.netmgt.flows.api.Flow;
import org.opennms.netmgt.flows.api.FlowSource;
import org.opennms.netmgt.flows.classification.ClassificationEngine;
import org.opennms.netmgt.flows.classification.ClassificationRequest;
import org.opennms.netmgt.flows.classification.persistence.api.Protocols;
import org.opennms.netmgt.flows.processing.enrichment.EnrichedFlow;
import org.opennms.netmgt.flows.processing.enrichment.NodeInfo;
import org.opennms.netmgt.flows.processing.impl.DocumentMangler;
import org.opennms.netmgt.model.OnmsCategory;
import org.opennms.netmgt.model.OnmsIpInterface;
import org.opennms.netmgt.model.OnmsNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DocumentEnricherImpl {
    private static final Logger LOG = LoggerFactory.getLogger(DocumentEnricherImpl.class);
    private static final String NODE_METADATA_CACHE = "flows.node.metadata";
    private final NodeDao nodeDao;
    private final IpInterfaceDao ipInterfaceDao;
    private final InterfaceToNodeCache interfaceToNodeCache;
    private final SessionUtils sessionUtils;
    private final ClassificationEngine classificationEngine;
    private final Cache<InterfaceToNodeCache.Entry, Optional<NodeInfo>> nodeInfoCache;
    private final Cache<NodeMetadataKey, Optional<NodeInfo>> nodeMetadataCache;
    private final Timer nodeLoadTimer;
    private final long clockSkewCorrectionThreshold;
    private final DocumentMangler mangler;

    public DocumentEnricherImpl(MetricRegistry metricRegistry, NodeDao nodeDao, IpInterfaceDao ipInterfaceDao, InterfaceToNodeCache interfaceToNodeCache, SessionUtils sessionUtils, ClassificationEngine classificationEngine, CacheConfig cacheConfig, long clockSkewCorrectionThreshold, DocumentMangler mangler) {
        this.nodeDao = Objects.requireNonNull(nodeDao);
        this.ipInterfaceDao = Objects.requireNonNull(ipInterfaceDao);
        this.interfaceToNodeCache = Objects.requireNonNull(interfaceToNodeCache);
        this.sessionUtils = Objects.requireNonNull(sessionUtils);
        this.classificationEngine = Objects.requireNonNull(classificationEngine);
        this.nodeInfoCache = new CacheBuilder().withConfig(cacheConfig).withCacheLoader((CacheLoader)new CacheLoader<InterfaceToNodeCache.Entry, Optional<NodeInfo>>(){

            public Optional<NodeInfo> load(InterfaceToNodeCache.Entry entry) {
                return DocumentEnricherImpl.this.getNodeInfo(entry);
            }
        }).build();
        CacheConfig nodeMetadataCacheConfig = this.buildMetadataCacheConfig(cacheConfig);
        this.nodeMetadataCache = new CacheBuilder().withConfig(nodeMetadataCacheConfig).withCacheLoader((CacheLoader)new CacheLoader<NodeMetadataKey, Optional<NodeInfo>>(){

            public Optional<NodeInfo> load(NodeMetadataKey key) {
                return DocumentEnricherImpl.this.getNodeInfoFromMetadataContext(key.contextKey, key.value);
            }
        }).build();
        this.nodeLoadTimer = metricRegistry.timer("nodeLoadTime");
        this.clockSkewCorrectionThreshold = clockSkewCorrectionThreshold;
        this.mangler = Objects.requireNonNull(mangler);
    }

    public List<EnrichedFlow> enrich(Collection<Flow> flows, FlowSource source) {
        if (flows.isEmpty()) {
            LOG.info("Nothing to enrich.");
            return Collections.emptyList();
        }
        return (List)this.sessionUtils.withTransaction(() -> flows.stream().flatMap(flow -> {
            Duration skew;
            EnrichedFlow document = this.mangler.mangle(EnrichedFlow.from(flow));
            if (document == null) {
                return Stream.empty();
            }
            document.setHost(source.getSourceAddress());
            document.setLocation(source.getLocation());
            this.getNodeInfoFromCache(source.getLocation(), source.getSourceAddress(), source.getContextKey(), flow.getNodeIdentifier()).ifPresent(document::setExporterNodeInfo);
            if (flow.getDstAddr() != null) {
                this.getNodeInfoFromCache(source.getLocation(), flow.getDstAddr(), null, null).ifPresent(document::setSrcNodeInfo);
            }
            if (flow.getSrcAddr() != null) {
                this.getNodeInfoFromCache(source.getLocation(), flow.getSrcAddr(), null, null).ifPresent(document::setDstNodeInfo);
            }
            if (flow.getSrcAddr() != null) {
                document.setSrcLocality(DocumentEnricherImpl.isPrivateAddress(flow.getSrcAddr()) ? Flow.Locality.PRIVATE : Flow.Locality.PUBLIC);
            }
            if (flow.getDstAddr() != null) {
                document.setDstLocality(DocumentEnricherImpl.isPrivateAddress(flow.getDstAddr()) ? Flow.Locality.PRIVATE : Flow.Locality.PUBLIC);
            }
            if (Flow.Locality.PUBLIC.equals((Object)document.getDstLocality()) || Flow.Locality.PUBLIC.equals((Object)document.getSrcLocality())) {
                document.setFlowLocality(Flow.Locality.PUBLIC);
            } else if (Flow.Locality.PRIVATE.equals((Object)document.getDstLocality()) || Flow.Locality.PRIVATE.equals((Object)document.getSrcLocality())) {
                document.setFlowLocality(Flow.Locality.PRIVATE);
            }
            ClassificationRequest classificationRequest = DocumentEnricherImpl.createClassificationRequest(document);
            if (classificationRequest.isClassifiable()) {
                document.setApplication(this.classificationEngine.classify(classificationRequest));
            }
            if (this.clockSkewCorrectionThreshold > 0L && (skew = Duration.between(flow.getReceivedAt(), flow.getTimestamp())).abs().toMillis() >= this.clockSkewCorrectionThreshold) {
                document.setClockCorrection(skew.negated());
                document.setTimestamp(flow.getTimestamp().minus(skew));
                document.setFirstSwitched(flow.getFirstSwitched().minus(skew));
                document.setDeltaSwitched(flow.getDeltaSwitched().minus(skew));
                document.setLastSwitched(flow.getLastSwitched().minus(skew));
            }
            return Stream.of(document);
        }).collect(Collectors.toList()));
    }

    private static boolean isPrivateAddress(String ipAddress) {
        InetAddress inetAddress = InetAddressUtils.addr((String)ipAddress);
        return inetAddress.isLoopbackAddress() || inetAddress.isLinkLocalAddress() || inetAddress.isSiteLocalAddress();
    }

    private Optional<NodeInfo> getNodeInfoFromCache(String location, String ipAddress, ContextKey contextKey, String value) {
        Optional entry;
        Optional nodeDocument = Optional.empty();
        if (contextKey != null && !Strings.isNullOrEmpty((String)value)) {
            NodeMetadataKey metadataKey = new NodeMetadataKey(contextKey, value);
            try {
                nodeDocument = (Optional)this.nodeMetadataCache.get((Object)metadataKey);
            }
            catch (ExecutionException e) {
                LOG.error("Error while retrieving NodeDocument from NodeMetadataCache: {}.", (Object)e.getMessage(), (Object)e);
                throw new RuntimeException(e);
            }
            if (nodeDocument.isPresent()) {
                return nodeDocument;
            }
        }
        if ((entry = this.interfaceToNodeCache.getFirst(location, InetAddressUtils.addr((String)ipAddress))).isPresent()) {
            try {
                return (Optional)this.nodeInfoCache.get((Object)((InterfaceToNodeCache.Entry)entry.get()));
            }
            catch (ExecutionException e) {
                LOG.error("Error while retrieving NodeDocument from NodeInfoCache: {}.", (Object)e.getMessage(), (Object)e);
                throw new RuntimeException(e);
            }
        }
        return nodeDocument;
    }

    private Optional<NodeInfo> getNodeInfoFromMetadataContext(ContextKey contextKey, String value) {
        List nodes;
        List ifaces;
        try (Timer.Context ctx = this.nodeLoadTimer.time();){
            ifaces = this.ipInterfaceDao.findInterfacesWithMetadata(contextKey.getContext(), contextKey.getKey(), value);
        }
        if (!ifaces.isEmpty()) {
            OnmsIpInterface iface = (OnmsIpInterface)ifaces.get(0);
            return this.mapOnmsNodeToNodeDocument(iface.getNode(), iface.getId());
        }
        try (Timer.Context ctx = this.nodeLoadTimer.time();){
            nodes = this.nodeDao.findNodeWithMetaData(contextKey.getContext(), contextKey.getKey(), value);
        }
        if (!nodes.isEmpty()) {
            OnmsNode node = (OnmsNode)nodes.get(0);
            return this.mapOnmsNodeToNodeDocument(node, node.getPrimaryInterface().getId());
        }
        return Optional.empty();
    }

    private Optional<NodeInfo> getNodeInfo(InterfaceToNodeCache.Entry entry) {
        OnmsNode onmsNode;
        try (Timer.Context ctx = this.nodeLoadTimer.time();){
            onmsNode = (OnmsNode)this.nodeDao.get((Serializable)Integer.valueOf(entry.nodeId));
        }
        return this.mapOnmsNodeToNodeDocument(onmsNode, entry.interfaceId);
    }

    private Optional<NodeInfo> mapOnmsNodeToNodeDocument(OnmsNode onmsNode, int interfaceId) {
        if (onmsNode != null) {
            NodeInfo nodeDocument = new NodeInfo();
            nodeDocument.setForeignSource(onmsNode.getForeignSource());
            nodeDocument.setForeignId(onmsNode.getForeignId());
            nodeDocument.setNodeId(onmsNode.getId());
            nodeDocument.setInterfaceId(interfaceId);
            nodeDocument.setCategories(onmsNode.getCategories().stream().map(OnmsCategory::getName).collect(Collectors.toList()));
            return Optional.of(nodeDocument);
        }
        return Optional.empty();
    }

    public static ClassificationRequest createClassificationRequest(EnrichedFlow document) {
        ClassificationRequest request = new ClassificationRequest();
        request.setProtocol(Protocols.getProtocol((Integer)document.getProtocol()));
        request.setLocation(document.getLocation());
        request.setExporterAddress(document.getHost());
        request.setDstAddress(document.getDstAddr());
        request.setDstPort(document.getDstPort());
        request.setSrcAddress(document.getSrcAddr());
        request.setSrcPort(document.getSrcPort());
        return request;
    }

    private CacheConfig buildMetadataCacheConfig(CacheConfig cacheConfig) {
        CacheConfig metadataCacheConfig = new CacheConfigBuilder().withName(NODE_METADATA_CACHE).withMaximumSize(cacheConfig.getMaximumSize().longValue()).withExpireAfterWrite(cacheConfig.getExpireAfterWrite().longValue()).build();
        cacheConfig.setRecordStats(true);
        cacheConfig.setMetricRegistry(cacheConfig.getMetricRegistry());
        return metadataCacheConfig;
    }

    private static class NodeMetadataKey {
        public final ContextKey contextKey;
        public final String value;

        private NodeMetadataKey(ContextKey contextKey, String value) {
            this.contextKey = contextKey;
            this.value = value;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            NodeMetadataKey that = (NodeMetadataKey)o;
            return Objects.equals(this.contextKey, that.contextKey) && Objects.equals(this.value, that.value);
        }

        public int hashCode() {
            return Objects.hash(this.contextKey, this.value);
        }
    }
}

