/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.netmgt.telemetry.protocols.bmp.adapter;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import java.net.InetAddress;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.opennms.core.utils.InetAddressUtils;
import org.opennms.netmgt.telemetry.api.adapter.TelemetryMessageLog;
import org.opennms.netmgt.telemetry.api.adapter.TelemetryMessageLogEntry;
import org.opennms.netmgt.telemetry.listeners.utils.BufferUtils;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.BmpAdapterTools;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.openbmp.BmpMessageHandler;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.openbmp.Context;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.openbmp.proto.AddressFamilyIdentifier;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.openbmp.proto.Message;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.openbmp.proto.Record;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.openbmp.proto.SubsequentAddressFamilyIdentifier;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.openbmp.proto.Type;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.openbmp.proto.records.BaseAttribute;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.openbmp.proto.records.Collector;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.openbmp.proto.records.Peer;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.openbmp.proto.records.Router;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.openbmp.proto.records.Stat;
import org.opennms.netmgt.telemetry.protocols.bmp.adapter.openbmp.proto.records.UnicastPrefix;
import org.opennms.netmgt.telemetry.protocols.bmp.transport.Transport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BmpAdapterCommon {
    private static final Logger LOG = LoggerFactory.getLogger(BmpAdapterCommon.class);

    public static void handleBmpMessage(TelemetryMessageLogEntry messageLogEntry, TelemetryMessageLog messageLog, BmpMessageHandler messageHandler, AtomicLong sequence) {
        Transport.Message message;
        LOG.trace("Parsing packet: {}", (Object)messageLogEntry);
        try {
            message = Transport.Message.parseFrom((byte[])messageLogEntry.getByteArray());
        }
        catch (InvalidProtocolBufferException e) {
            LOG.error("Invalid message", (Throwable)e);
            return;
        }
        String collectorHashId = Record.hash(messageLog.getSystemId());
        String routerHashId = Record.hash(messageLog.getSourceAddress());
        Context context = new Context(messageLog.getSystemId(), collectorHashId, routerHashId, Instant.ofEpochMilli(messageLogEntry.getTimestamp()), InetAddressUtils.addr((String)messageLog.getSourceAddress()), messageLog.getSourcePort(), messageLog.getLocation());
        switch (message.getPacketCase()) {
            case HEARTBEAT: {
                BmpAdapterCommon.handleHeartbeatMessage(messageHandler, message, message.getHeartbeat(), context, sequence);
                break;
            }
            case INITIATION: {
                BmpAdapterCommon.handleInitiationMessage(messageHandler, message, message.getInitiation(), context, sequence);
                break;
            }
            case TERMINATION: {
                BmpAdapterCommon.handleTerminationMessage(messageHandler, message, message.getTermination(), context, sequence);
                break;
            }
            case PEER_UP: {
                BmpAdapterCommon.handlePeerUpNotification(messageHandler, message, message.getPeerUp(), context, sequence);
                break;
            }
            case PEER_DOWN: {
                BmpAdapterCommon.handlePeerDownNotification(messageHandler, message, message.getPeerDown(), context, sequence);
                break;
            }
            case STATISTICS_REPORT: {
                BmpAdapterCommon.handleStatisticReport(messageHandler, message, message.getStatisticsReport(), context, sequence);
                break;
            }
            case ROUTE_MONITORING: {
                BmpAdapterCommon.handleRouteMonitoringMessage(messageHandler, message, message.getRouteMonitoring(), context, sequence);
                break;
            }
        }
    }

    public static void handleHeartbeatMessage(BmpMessageHandler messageHandler, Transport.Message message, Transport.Heartbeat heartbeat, Context context, AtomicLong sequence) {
        Collector collector = new Collector();
        switch (heartbeat.getMode()) {
            case STARTED: {
                collector.action = Collector.Action.STARTED;
                break;
            }
            case STOPPED: {
                collector.action = Collector.Action.STOPPED;
                break;
            }
            case PERIODIC: {
                collector.action = Collector.Action.HEARTBEAT;
                break;
            }
            case CHANGE: {
                collector.action = Collector.Action.CHANGE;
            }
        }
        collector.sequence = sequence.getAndIncrement();
        collector.adminId = context.adminId;
        collector.hash = context.collectorHashId;
        collector.routers = Lists.transform((List)heartbeat.getRoutersList(), BmpAdapterTools::address);
        collector.timestamp = context.timestamp;
        messageHandler.handle(new Message(context.collectorHashId, Type.COLLECTOR, (List<Record>)ImmutableList.of((Object)collector)), context);
    }

    public static void handleInitiationMessage(BmpMessageHandler messageHandler, Transport.Message message, Transport.InitiationPacket initiation, Context context, AtomicLong sequence) {
        Router router = new Router();
        router.action = Router.Action.INIT;
        router.sequence = sequence.getAndIncrement();
        router.name = initiation.getSysName() != null ? initiation.getSysName() : initiation.getHostname();
        router.hash = context.routerHashId;
        router.ipAddress = context.sourceAddress;
        router.description = Joiner.on((char)'\n').join((Iterable)initiation.getSysDescList());
        router.termCode = null;
        router.termReason = null;
        router.initData = Joiner.on((char)'\n').join((Iterable)initiation.getMessageList());
        router.termData = null;
        router.timestamp = context.timestamp;
        router.bgpId = initiation.hasBgpId() ? BmpAdapterTools.address(initiation.getBgpId()) : null;
        messageHandler.handle(new Message(context.collectorHashId, Type.ROUTER, (List<Record>)ImmutableList.of((Object)router)), context);
    }

    public static void handleTerminationMessage(BmpMessageHandler messageHandler, Transport.Message message, Transport.TerminationPacket termination, Context context, AtomicLong sequence) {
        Router router = new Router();
        router.action = Router.Action.TERM;
        router.sequence = sequence.getAndIncrement();
        router.name = null;
        router.hash = context.routerHashId;
        router.ipAddress = context.sourceAddress;
        router.description = null;
        router.termCode = termination.getReason();
        switch (router.termCode) {
            case 0: {
                router.termReason = "Session administratively closed. The session might be re-initiated";
                break;
            }
            case 1: {
                router.termReason = "Unspecified reason";
                break;
            }
            case 2: {
                router.termReason = "Out of resources. The router has exhausted resources available for the BMP session";
                break;
            }
            case 3: {
                router.termReason = "Redundant connection. The router has determined that this connection is redundant with another one";
                break;
            }
            case 4: {
                router.termReason = "Session permanently administratively closed, will not be re-initiated";
                break;
            }
            default: {
                router.termReason = "Unknown reason";
            }
        }
        router.initData = null;
        router.termData = Joiner.on((char)'\n').join((Iterable)termination.getMessageList());
        router.timestamp = context.timestamp;
        router.bgpId = null;
        messageHandler.handle(new Message(context.collectorHashId, Type.ROUTER, (List<Record>)ImmutableList.of((Object)router)), context);
    }

    public static void handlePeerUpNotification(BmpMessageHandler messageHandler, Transport.Message message, Transport.PeerUpPacket peerUp, Context context, AtomicLong sequence) {
        Transport.Peer bgpPeer = peerUp.getPeer();
        Peer peer = new Peer();
        peer.action = Peer.Action.UP;
        peer.sequence = sequence.getAndIncrement();
        peer.name = !Strings.isNullOrEmpty((String)bgpPeer.getHostname()) ? bgpPeer.getHostname() : BmpAdapterTools.addressAsStr(bgpPeer.getAddress());
        peer.hash = Record.hash(bgpPeer.getAddress(), bgpPeer.getDistinguisher(), context.routerHashId);
        peer.routerHash = context.routerHashId;
        peer.remoteBgpId = BmpAdapterTools.address(peerUp.getRecvMsg().getId());
        peer.routerIp = context.sourceAddress;
        peer.timestamp = BmpAdapterTools.timestamp(bgpPeer.getTimestamp());
        peer.remoteAsn = BmpAdapterTools.uint32(peerUp.getRecvMsg().getAs());
        peer.remoteIp = BmpAdapterTools.address(bgpPeer.getAddress());
        peer.peerRd = BmpAdapterTools.asAttr((int)bgpPeer.getDistinguisher());
        peer.remotePort = peerUp.getRemotePort();
        peer.localAsn = BmpAdapterTools.uint32(peerUp.getSendMsg().getAs());
        peer.localIp = BmpAdapterTools.address(peerUp.getLocalAddress());
        peer.localPort = peerUp.getLocalPort();
        peer.localBgpId = BmpAdapterTools.address(peerUp.getSendMsg().getId());
        peer.infoData = peerUp.getMessage();
        peer.advertisedCapabilities = peerUp.getSendMsg().getCapabilities().getCapabilityList().stream().map(c -> BmpAdapterCommon.generateCapabilityMessage(c.getCode(), c.getValue())).collect(Collectors.joining(","));
        peer.receivedCapabilities = peerUp.getRecvMsg().getCapabilities().getCapabilityList().stream().map(c -> BmpAdapterCommon.generateCapabilityMessage(c.getCode(), c.getValue())).collect(Collectors.joining(","));
        peer.remoteHolddown = BmpAdapterTools.uint32(peerUp.getRecvMsg().getHoldTime());
        peer.advertisedHolddown = BmpAdapterTools.uint32(peerUp.getSendMsg().getHoldTime());
        peer.bmpReason = null;
        peer.bgpErrorCode = null;
        peer.bgpErrorSubcode = null;
        peer.errorText = null;
        peer.l3vpn = bgpPeer.getType() == Transport.Peer.Type.RD_INSTANCE;
        peer.prePolicy = bgpPeer.hasPeerFlags() && bgpPeer.getPeerFlags().getPolicy() == Transport.Peer.PeerFlags.Policy.PRE_POLICY;
        peer.ipv4 = BmpAdapterTools.isV4(bgpPeer.getAddress());
        peer.locRib = bgpPeer.getType() == Transport.Peer.Type.LOC_RIB_INSTANCE;
        peer.locRibFiltered = bgpPeer.hasLocRibFlags() && bgpPeer.getLocRibFlags().getFiltered();
        peer.tableName = peerUp.getTableName();
        messageHandler.handle(new Message(context.collectorHashId, Type.PEER, (List<Record>)ImmutableList.of((Object)peer)), context);
    }

    public static void handlePeerDownNotification(BmpMessageHandler messageHandler, Transport.Message message, Transport.PeerDownPacket peerDown, Context context, AtomicLong sequence) {
        Transport.Peer bgpPeer = peerDown.getPeer();
        Peer peer = new Peer();
        peer.action = Peer.Action.DOWN;
        peer.sequence = sequence.getAndIncrement();
        peer.name = !Strings.isNullOrEmpty((String)bgpPeer.getHostname()) ? bgpPeer.getHostname() : BmpAdapterTools.addressAsStr(bgpPeer.getAddress());
        peer.hash = Record.hash(bgpPeer.getAddress(), bgpPeer.getDistinguisher(), context.routerHashId);
        peer.routerHash = context.routerHashId;
        peer.remoteBgpId = BmpAdapterTools.address(bgpPeer.getId());
        peer.routerIp = context.sourceAddress;
        peer.timestamp = BmpAdapterTools.timestamp(bgpPeer.getTimestamp());
        peer.remoteAsn = BmpAdapterTools.uint32(bgpPeer.getAs());
        peer.remoteIp = BmpAdapterTools.address(bgpPeer.getAddress());
        peer.peerRd = BmpAdapterTools.asAttr((int)bgpPeer.getDistinguisher());
        peer.remotePort = null;
        peer.localAsn = null;
        peer.localIp = null;
        peer.localPort = null;
        peer.localBgpId = null;
        peer.infoData = null;
        peer.advertisedCapabilities = null;
        peer.receivedCapabilities = null;
        peer.remoteHolddown = null;
        peer.advertisedHolddown = null;
        peer.bmpReason = peerDown.getReasonCase().getNumber() - 1;
        switch (peerDown.getReasonCase().getNumber()) {
            case 2: {
                peer.bgpErrorCode = peerDown.getLocalBgpNotification().getCode();
                peer.bgpErrorSubcode = peerDown.getLocalBgpNotification().getSubcode();
                break;
            }
            case 4: {
                peer.bgpErrorCode = peerDown.getRemoteBgpNotification().getCode();
                peer.bgpErrorSubcode = peerDown.getRemoteBgpNotification().getSubcode();
                break;
            }
            default: {
                peer.bgpErrorCode = null;
                peer.bgpErrorSubcode = null;
            }
        }
        if (peer.bgpErrorCode != null && peer.bgpErrorSubcode != null) {
            peer.errorText = Error.from(peer.bgpErrorCode, peer.bgpErrorSubcode).getErrorText();
        }
        peer.l3vpn = bgpPeer.getType() == Transport.Peer.Type.RD_INSTANCE;
        peer.prePolicy = bgpPeer.hasPeerFlags() && bgpPeer.getPeerFlags().getPolicy() == Transport.Peer.PeerFlags.Policy.PRE_POLICY;
        peer.ipv4 = BmpAdapterTools.isV4(bgpPeer.getAddress());
        peer.locRib = bgpPeer.getType() == Transport.Peer.Type.LOC_RIB_INSTANCE;
        peer.locRibFiltered = bgpPeer.hasLocRibFlags() && bgpPeer.getLocRibFlags().getFiltered();
        peer.tableName = "";
        messageHandler.handle(new Message(context.collectorHashId, Type.PEER, (List<Record>)ImmutableList.of((Object)peer)), context);
    }

    public static void handleStatisticReport(BmpMessageHandler messageHandler, Transport.Message message, Transport.StatisticsReportPacket statisticsReport, Context context, AtomicLong sequence) {
        Transport.Peer peer = statisticsReport.getPeer();
        Stat stat = new Stat();
        stat.action = Stat.Action.ADD;
        stat.sequence = sequence.getAndIncrement();
        stat.routerHash = Record.hash(context.sourceAddress.getHostAddress(), context.collectorHashId);
        stat.routerIp = context.sourceAddress;
        stat.peerHash = Record.hash(peer.getAddress(), peer.getDistinguisher(), stat.routerHash);
        stat.peerIp = BmpAdapterTools.address(peer.getAddress());
        stat.peerAsn = BmpAdapterTools.uint32(peer.getAs());
        stat.timestamp = BmpAdapterTools.timestamp(peer.getTimestamp());
        stat.prefixesRejected = statisticsReport.getRejected().getCount();
        stat.knownDupPrefixes = statisticsReport.getDuplicatePrefix().getCount();
        stat.knownDupWithdraws = statisticsReport.getDuplicateWithdraw().getCount();
        stat.invalidClusterList = statisticsReport.getInvalidUpdateDueToClusterListLoop().getCount();
        stat.invalidAsPath = statisticsReport.getInvalidUpdateDueToAsPathLoop().getCount();
        stat.invalidOriginatorId = statisticsReport.getInvalidUpdateDueToOriginatorId().getCount();
        stat.invalidAsConfed = statisticsReport.getInvalidUpdateDueToAsConfedLoop().getCount();
        stat.prefixesPrePolicy = statisticsReport.getAdjRibIn().getValue();
        stat.prefixesPostPolicy = statisticsReport.getLocalRib().getValue();
        messageHandler.handle(new Message(context.collectorHashId, Type.BMP_STAT, (List<Record>)ImmutableList.of((Object)stat)), context);
    }

    private static BaseAttribute toBaseAttributeRecord(Transport.RouteMonitoringPacket routeMonitoring, Context context, AtomicLong sequence) {
        Transport.Peer peer = routeMonitoring.getPeer();
        BaseAttribute baseAttr = new BaseAttribute();
        baseAttr.action = BaseAttribute.Action.ADD;
        baseAttr.sequence = sequence.getAndIncrement();
        baseAttr.routerHash = context.getRouterHash();
        baseAttr.routerIp = context.sourceAddress;
        baseAttr.peerHash = Record.hash(peer.getAddress(), peer.getDistinguisher(), baseAttr.routerHash);
        baseAttr.peerIp = BmpAdapterTools.address(peer.getAddress());
        baseAttr.peerAsn = BmpAdapterTools.uint32(peer.getAs());
        baseAttr.timestamp = context.timestamp;
        baseAttr.origin = BmpAdapterTools.getPathAttributeOfType(routeMonitoring, Transport.RouteMonitoringPacket.PathAttribute.ValueCase.ORIGIN).map(attr -> attr.getOrigin().name().toLowerCase()).orElse("");
        baseAttr.asPathCount = 0;
        baseAttr.asPath = BmpAdapterTools.getPathAttributeOfType(routeMonitoring, Transport.RouteMonitoringPacket.PathAttribute.ValueCase.AS_PATH).map(asPathAttr -> {
            StringBuilder asPath = new StringBuilder();
            asPathAttr.getAsPath().getSegmentsList().forEach(segment -> {
                if (Transport.RouteMonitoringPacket.PathAttribute.AsPath.Segment.Type.AS_SET.equals((Object)segment.getType())) {
                    asPath.append("{");
                }
                segment.getPathsList().forEach(segmentPath -> {
                    asPath.append(segmentPath);
                    asPath.append(" ");
                    Integer n = baseAttr.asPathCount;
                    Integer n2 = baseAttr.asPathCount = Integer.valueOf(baseAttr.asPathCount + 1);
                    baseAttr.originAs = BmpAdapterTools.uint32(segmentPath);
                });
                if (Transport.RouteMonitoringPacket.PathAttribute.AsPath.Segment.Type.AS_SET.equals((Object)segment.getType())) {
                    asPath.append("}");
                }
            });
            return asPath.toString();
        }).orElse("");
        BmpAdapterTools.getPathAttributeOfType(routeMonitoring, Transport.RouteMonitoringPacket.PathAttribute.ValueCase.NEXT_HOP).map(attr -> attr.getNextHop().getAddress()).ifPresent(nextHop -> {
            baseAttr.nextHop = BmpAdapterTools.address(nextHop);
            baseAttr.nextHopIpv4 = BmpAdapterTools.isV4(nextHop);
        });
        BmpAdapterTools.getPathAttributeOfType(routeMonitoring, Transport.RouteMonitoringPacket.PathAttribute.ValueCase.MULTI_EXIT_DISC).map(attr -> attr.getMultiExitDisc().getDiscriminator()).ifPresent(med -> {
            baseAttr.med = BmpAdapterTools.uint32(med);
        });
        BmpAdapterTools.getPathAttributeOfType(routeMonitoring, Transport.RouteMonitoringPacket.PathAttribute.ValueCase.LOCAL_PREF).map(attr -> attr.getLocalPref().getPreference()).ifPresent(localPref -> {
            baseAttr.localPref = BmpAdapterTools.uint32(localPref);
        });
        BmpAdapterTools.getPathAttributeOfType(routeMonitoring, Transport.RouteMonitoringPacket.PathAttribute.ValueCase.AGGREGATOR).map(Transport.RouteMonitoringPacket.PathAttribute::getAggregator).ifPresent(agg -> {
            baseAttr.aggregator = String.format("%d %s", agg.getAs(), BmpAdapterTools.addressAsStr(agg.getAddress()));
        });
        baseAttr.communityList = BmpAdapterTools.getPathAttributesOfType(routeMonitoring, Transport.RouteMonitoringPacket.PathAttribute.ValueCase.COMMUNITY).map(Transport.RouteMonitoringPacket.PathAttribute::getCommunity).map(BmpAdapterTools::asAttr).collect(Collectors.joining(" "));
        baseAttr.extCommunityList = Stream.concat(BmpAdapterTools.getPathAttributesOfType(routeMonitoring, Transport.RouteMonitoringPacket.PathAttribute.ValueCase.EXTENDED_COMMUNITIES).map(Transport.RouteMonitoringPacket.PathAttribute::getExtendedCommunities).flatMap(extendedCommunities -> extendedCommunities.getExtendedCommunitiesList().stream().map(extendedCommunity -> String.format("%s=%s", extendedCommunity.getType(), extendedCommunity.getValue()))), BmpAdapterTools.getPathAttributesOfType(routeMonitoring, Transport.RouteMonitoringPacket.PathAttribute.ValueCase.EXTENDED_V6_COMMUNITIES).map(Transport.RouteMonitoringPacket.PathAttribute::getExtendedV6Communities).flatMap(extendedV6Communities -> extendedV6Communities.getExtendedCommunitiesList().stream().map(extendedCommunity -> String.format("%s=%s", extendedCommunity.getType(), extendedCommunity.getValue())))).collect(Collectors.joining(" "));
        BmpAdapterTools.getPathAttributeOfType(routeMonitoring, Transport.RouteMonitoringPacket.PathAttribute.ValueCase.CLUSTER_LIST).map(Transport.RouteMonitoringPacket.PathAttribute::getClusterList).ifPresent(clusterList -> {
            baseAttr.clusterList = clusterList.getClusterIdList().stream().map(BmpAdapterTools::addressAsStr).collect(Collectors.joining(" "));
        });
        BmpAdapterTools.getPathAttributeOfType(routeMonitoring, Transport.RouteMonitoringPacket.PathAttribute.ValueCase.LARGE_COMMUNITIES).map(Transport.RouteMonitoringPacket.PathAttribute::getLargeCommunities).ifPresent(largeCommunities -> {
            baseAttr.largeCommunityList = largeCommunities.getLargeCommunitiesList().stream().map(largeCommunity -> String.format("%d:%d:%d", BmpAdapterTools.uint32(largeCommunity.getGlobalAdministrator()), BmpAdapterTools.uint32(largeCommunity.getLocalDataPart1()), BmpAdapterTools.uint32(largeCommunity.getLocalDataPart2()))).collect(Collectors.joining(" "));
        });
        BmpAdapterTools.getPathAttributeOfType(routeMonitoring, Transport.RouteMonitoringPacket.PathAttribute.ValueCase.ORIGINATOR_ID).map(Transport.RouteMonitoringPacket.PathAttribute::getOriginatorId).ifPresent(originatorId -> {
            baseAttr.originatorId = Long.toString(BmpAdapterTools.uint32(originatorId));
        });
        baseAttr.atomicAgg = BmpAdapterTools.getPathAttributeOfType(routeMonitoring, Transport.RouteMonitoringPacket.PathAttribute.ValueCase.ATOMIC_AGGREGATE).isPresent();
        baseAttr.hash = Record.hash(baseAttr.asPath, Record.nullSafeStr(baseAttr.nextHop), baseAttr.aggregator, baseAttr.origin, Record.nullSafeStr(baseAttr.med), Record.nullSafeStr(baseAttr.localPref), baseAttr.communityList, baseAttr.extCommunityList, baseAttr.peerHash);
        return baseAttr;
    }

    private static UnicastPrefix toUnicastPrefixRecord(Transport.RouteMonitoringPacket routeMonitoring, Transport.RouteMonitoringPacket.Route route, BaseAttribute baseAttr, Context context, AtomicLong sequence) {
        Transport.Peer peer = routeMonitoring.getPeer();
        UnicastPrefix unicastPrefix = new UnicastPrefix();
        unicastPrefix.sequence = sequence.incrementAndGet();
        unicastPrefix.routerHash = context.getRouterHash();
        unicastPrefix.routerIp = context.sourceAddress;
        unicastPrefix.peerHash = Record.hash(peer.getAddress(), peer.getDistinguisher(), unicastPrefix.routerHash);
        unicastPrefix.peerIp = BmpAdapterTools.address(peer.getAddress());
        unicastPrefix.peerAsn = BmpAdapterTools.uint32(peer.getAs());
        unicastPrefix.timestamp = context.timestamp;
        unicastPrefix.prefix = BmpAdapterTools.address(route.getPrefix());
        unicastPrefix.length = route.getLength();
        unicastPrefix.ipv4 = BmpAdapterTools.isV4(route.getPrefix());
        unicastPrefix.pathId = route.getPathId();
        unicastPrefix.labels = route.getLabels();
        unicastPrefix.prePolicy = peer.hasPeerFlags() && peer.getPeerFlags().getPolicy() == Transport.Peer.PeerFlags.Policy.PRE_POLICY;
        boolean bl = unicastPrefix.adjIn = peer.hasPeerFlags() && peer.getPeerFlags().getAdjIn();
        if (baseAttr != null) {
            unicastPrefix.baseAttrHash = baseAttr.hash;
            unicastPrefix.origin = baseAttr.origin;
            unicastPrefix.asPath = baseAttr.asPath;
            unicastPrefix.asPathCount = baseAttr.asPathCount;
            unicastPrefix.originAs = baseAttr.originAs;
            unicastPrefix.nextHop = baseAttr.nextHop;
            unicastPrefix.med = baseAttr.med;
            unicastPrefix.localPref = baseAttr.localPref;
            unicastPrefix.aggregator = baseAttr.aggregator;
            unicastPrefix.communityList = baseAttr.communityList;
            unicastPrefix.extCommunityList = baseAttr.extCommunityList;
            unicastPrefix.clusterList = baseAttr.clusterList;
            unicastPrefix.atomicAgg = baseAttr.atomicAgg;
            unicastPrefix.nextHopIpv4 = baseAttr.nextHopIpv4;
            unicastPrefix.originatorId = baseAttr.originatorId;
            unicastPrefix.largeCommunityList = baseAttr.largeCommunityList;
        }
        unicastPrefix.hash = Record.hash(InetAddressUtils.str((InetAddress)unicastPrefix.prefix), Integer.toString(unicastPrefix.length), unicastPrefix.peerHash, Long.toString(unicastPrefix.pathId), Strings.isNullOrEmpty((String)unicastPrefix.labels) ? "0" : "1");
        return unicastPrefix;
    }

    static void handleRouteMonitoringMessage(BmpMessageHandler messageHandler, Transport.Message message, Transport.RouteMonitoringPacket routeMonitoring, Context context, AtomicLong sequence) {
        BaseAttribute baseAttr;
        ArrayList<Record> unicastPrefixRecords = new ArrayList<Record>(routeMonitoring.getWithdrawsCount() + routeMonitoring.getReachablesCount());
        for (Object route : routeMonitoring.getWithdrawsList()) {
            UnicastPrefix unicastPrefix = BmpAdapterCommon.toUnicastPrefixRecord(routeMonitoring, (Transport.RouteMonitoringPacket.Route)route, null, context, sequence);
            unicastPrefix.action = UnicastPrefix.Action.DELETE;
            unicastPrefixRecords.add(unicastPrefix);
        }
        if (routeMonitoring.getReachablesCount() > 0) {
            baseAttr = BmpAdapterCommon.toBaseAttributeRecord(routeMonitoring, context, sequence);
            for (Transport.RouteMonitoringPacket.Route route : routeMonitoring.getReachablesList()) {
                UnicastPrefix unicastPrefix = BmpAdapterCommon.toUnicastPrefixRecord(routeMonitoring, route, baseAttr, context, sequence);
                unicastPrefix.action = UnicastPrefix.Action.ADD;
                unicastPrefixRecords.add(unicastPrefix);
            }
            List mpReachNlriList = routeMonitoring.getAttributesList().stream().filter(a -> a.hasMpReachNrli()).map(a -> a.getMpReachNrli()).collect(Collectors.toList());
            for (Transport.RouteMonitoringPacket.PathAttribute.MultiprotocolReachableNrli multiprotocolReachableNrli : mpReachNlriList) {
                UnicastPrefix unicastPrefix;
                for (Transport.RouteMonitoringPacket.Route route : multiprotocolReachableNrli.getAdvertisedList()) {
                    unicastPrefix = BmpAdapterCommon.toUnicastPrefixRecord(routeMonitoring, route, baseAttr, context, sequence);
                    unicastPrefix.action = UnicastPrefix.Action.ADD;
                    unicastPrefixRecords.add(unicastPrefix);
                }
                for (Transport.RouteMonitoringPacket.Route route : multiprotocolReachableNrli.getVpnAdvertisedList()) {
                    unicastPrefix = BmpAdapterCommon.toUnicastPrefixRecord(routeMonitoring, route, baseAttr, context, sequence);
                    unicastPrefix.action = UnicastPrefix.Action.ADD;
                    unicastPrefixRecords.add(unicastPrefix);
                }
            }
        } else {
            baseAttr = null;
        }
        if (baseAttr != null) {
            messageHandler.handle(new Message(context.collectorHashId, Type.BASE_ATTRIBUTE, (List<Record>)ImmutableList.of((Object)baseAttr)), context);
        }
        messageHandler.handle(new Message(context.collectorHashId, Type.UNICAST_PREFIX, unicastPrefixRecords), context);
    }

    private static String generateCapabilityMessage(int code, ByteString byteString) {
        ByteBuf value = Unpooled.wrappedBuffer((byte[])byteString.toByteArray());
        if (code == 72 || code >= 10 && code <= 63 || code >= 74 && code <= 127) {
            return String.format("Unassigned (%d)", code);
        }
        if (code >= 128 && code <= 255) {
            return String.format("Reserved for Private Use", code);
        }
        switch (code) {
            case 0: {
                return "Reserved";
            }
            case 1: {
                int afi = BufferUtils.uint16((ByteBuf)value);
                BufferUtils.skip((ByteBuf)value, (int)1);
                int safi = BufferUtils.uint8((ByteBuf)value);
                return String.format("Multiprotocol Extensions for BGP-4 (1): afi=%d safi=%d: %s %s", code, afi, safi, AddressFamilyIdentifier.from(afi).getDescription(), SubsequentAddressFamilyIdentifier.from(safi).getDescription());
            }
            case 2: {
                return "Route Refresh Capability for BGP-4 (2)";
            }
            case 3: {
                return "Outbound Route Filtering Capability (3)";
            }
            case 4: {
                return "Multiple routes to a destination capability (deprecated) (4)";
            }
            case 5: {
                return "Extended Next Hop Encoding (5)";
            }
            case 6: {
                return "BGP Extended Message (6)";
            }
            case 7: {
                return "BGPsec Capability (7)";
            }
            case 8: {
                return "BGP Role (TEMPORARY - registered 2018-03-29, extension registered 2019-03-18, expires 2020-03-29) (8)";
            }
            case 9: {
                return "Multiple Labels Capability (9)";
            }
            case 64: {
                return "Graceful Restart Capability (64)";
            }
            case 65: {
                return "Support for 4-octet AS number capability (65)";
            }
            case 66: {
                return "Deprecated (2003-03-06) (66)";
            }
            case 67: {
                return "Support for Dynamic Capability (capability specific) (67)";
            }
            case 68: {
                return "Multisession BGP Capability (68)";
            }
            case 69: {
                List strings = BufferUtils.repeatRemaining((ByteBuf)value, b -> {
                    int afi = BufferUtils.uint16((ByteBuf)b);
                    int safi = BufferUtils.uint8((ByteBuf)b);
                    int sendReceive = BufferUtils.uint8((ByteBuf)b);
                    return String.format("afi=%d safi=%d send/receive=%d: %s %s %s", new Object[]{afi, safi, sendReceive, AddressFamilyIdentifier.from(afi), SubsequentAddressFamilyIdentifier.from(safi), BmpAdapterCommon.parseSendReceive(sendReceive)});
                });
                return String.format("ADD-PATH Capability (69): %s", String.join((CharSequence)", ", strings));
            }
            case 70: {
                return "Enhanced Route Refresh Capability (70)";
            }
            case 71: {
                return "Long-Lived Graceful Restart (71)";
            }
            case 73: {
                return "FQDN Capability (73)";
            }
        }
        return String.format("Unknown capability (%d)", code);
    }

    private static String parseSendReceive(int sendReceive) {
        switch (sendReceive) {
            case 1: {
                return "Receive";
            }
            case 2: {
                return "Send";
            }
            case 3: {
                return "Send/Receive";
            }
        }
        return "unknown";
    }

    public static enum Error {
        CONNECTION_NOT_SYNCHRONIZED("Connection not synchronized"),
        BAD_MESSAGE_LENGTH("Bad message header length"),
        BAD_MESSAGE_TYPE("Bad message header type"),
        UNSUPPORTED_VERSION_NUMBER("Unsupported BGP version"),
        BAD_PEER_AS("Incorrect peer AS"),
        BAD_BGP_IDENTIFIER("Bad BGP ID"),
        UNSUPPORTED_OPTIONAL_PARAMETER("Unsupported optinal parameter"),
        AUTHENTICATION_FAILURE("Authentication failure"),
        UNACCEPTABLE_HOLD_TIME("Unacceptable hold time"),
        MALFORMED_ATTRIBUTE_LIST("Malformed attribute list"),
        UNRECOGNIZED_WELL_KNOWN_ATTRIBUTE("Unrecognized well known attribute"),
        MISSING_WELL_KNOWN_ATTRIBUTE("Missing well known attribute"),
        ATTRIBUTE_FLAGS_ERROR("Update attribute flags error"),
        ATTRIBUTE_LENGTH_ERROR("Update attribute length error"),
        INVALID_ORIGIN_ATTRIBUTE("Invalid origin"),
        ROUTING_LOOP("Routing loop"),
        INVALID_NEXT_HOP_ATTRIBUTE("Invalid next hop address/attribute"),
        OPTIONAL_ATTRIBUTE_ERROR("Update optional attribute error"),
        INVALID_NETWORK_FIELD("Invalid network field"),
        MALFORMED_AS_PATH("Malformed AS_PATH"),
        HOLD_TIMER_EXPIRED("Hold timer expired"),
        FSM_ERROR("FSM error"),
        MAXIMUM_NUMBER_OF_PREFIXES_REACHED("Maximum number of prefixes reached"),
        ADMINISTRATIVE_SHUTDOWN("Administrative shutdown"),
        PEER_DECONFIGURED("Peer de-configured"),
        ADMINISTRATIVE_RESET("Administrative reset"),
        CONNECTION_RESET("Connection rejected"),
        OTHER_CONFIGURATION_CHANGE("Other configuration change"),
        CONNECTION_COLLISION_RESOLUTION("Connection collision resolution"),
        OUT_OF_RESOURCES("Maximum number of prefixes reached"),
        UNKNOWN("Unknown notification type");

        private String errorText;

        private Error(String errorText) {
            this.errorText = errorText;
        }

        public String getErrorText() {
            return this.errorText;
        }

        public static Error from(int code, int subcode) {
            switch ((code << 8) + subcode) {
                case 257: {
                    return CONNECTION_NOT_SYNCHRONIZED;
                }
                case 258: {
                    return BAD_MESSAGE_LENGTH;
                }
                case 259: {
                    return BAD_MESSAGE_TYPE;
                }
                case 513: {
                    return UNSUPPORTED_VERSION_NUMBER;
                }
                case 514: {
                    return BAD_PEER_AS;
                }
                case 515: {
                    return BAD_BGP_IDENTIFIER;
                }
                case 516: {
                    return UNSUPPORTED_OPTIONAL_PARAMETER;
                }
                case 517: {
                    return AUTHENTICATION_FAILURE;
                }
                case 518: {
                    return UNACCEPTABLE_HOLD_TIME;
                }
                case 769: {
                    return MALFORMED_ATTRIBUTE_LIST;
                }
                case 770: {
                    return UNRECOGNIZED_WELL_KNOWN_ATTRIBUTE;
                }
                case 771: {
                    return MISSING_WELL_KNOWN_ATTRIBUTE;
                }
                case 772: {
                    return ATTRIBUTE_FLAGS_ERROR;
                }
                case 773: {
                    return ATTRIBUTE_LENGTH_ERROR;
                }
                case 774: {
                    return INVALID_ORIGIN_ATTRIBUTE;
                }
                case 775: {
                    return ROUTING_LOOP;
                }
                case 776: {
                    return INVALID_NEXT_HOP_ATTRIBUTE;
                }
                case 777: {
                    return OPTIONAL_ATTRIBUTE_ERROR;
                }
                case 778: {
                    return INVALID_NETWORK_FIELD;
                }
                case 779: {
                    return MALFORMED_AS_PATH;
                }
                case 1025: {
                    return HOLD_TIMER_EXPIRED;
                }
                case 1281: {
                    return FSM_ERROR;
                }
                case 1537: {
                    return MAXIMUM_NUMBER_OF_PREFIXES_REACHED;
                }
                case 1538: {
                    return ADMINISTRATIVE_SHUTDOWN;
                }
                case 1539: {
                    return PEER_DECONFIGURED;
                }
                case 1540: {
                    return ADMINISTRATIVE_RESET;
                }
                case 1541: {
                    return CONNECTION_RESET;
                }
                case 1542: {
                    return OTHER_CONFIGURATION_CHANGE;
                }
                case 1543: {
                    return CONNECTION_COLLISION_RESOLUTION;
                }
                case 1544: {
                    return OUT_OF_RESOURCES;
                }
            }
            return UNKNOWN;
        }
    }
}

